import jsPDF from 'jspdf';
import { Sale, ShopSettings } from '../types';
import { format } from 'date-fns';

interface ReceiptData {
  sale: Sale & {
    sale_items?: Array<{
      service?: { name: string };
      quantity: number;
      unit_price: number;
      total_price: number;
    }>;
  };
  shopSettings: ShopSettings | null;
  cashierName: string;
}

export const generateReceipt = async ({ sale, shopSettings, cashierName }: ReceiptData) => {
  const pdf = new jsPDF({
    orientation: 'portrait',
    unit: 'mm',
    format: [80, 200], // Thermal printer size
  });

  let yPosition = 10;
  const lineHeight = 4;
  const pageWidth = 80;

  // Helper function to add centered text
  const addCenteredText = (text: string, fontSize = 10, isBold = false) => {
    pdf.setFontSize(fontSize);
    if (isBold) {
      pdf.setFont('helvetica', 'bold');
    } else {
      pdf.setFont('helvetica', 'normal');
    }
    const textWidth = pdf.getTextWidth(text);
    const x = (pageWidth - textWidth) / 2;
    pdf.text(text, x, yPosition);
    yPosition += lineHeight;
  };

  // Helper function to add left-aligned text
  const addLeftText = (text: string, fontSize = 8) => {
    pdf.setFontSize(fontSize);
    pdf.setFont('helvetica', 'normal');
    pdf.text(text, 5, yPosition);
    yPosition += lineHeight;
  };

  // Helper function to add line
  const addLine = () => {
    pdf.line(5, yPosition, pageWidth - 5, yPosition);
    yPosition += lineHeight;
  };

  // Shop header
  if (shopSettings?.shop_name) {
    addCenteredText(shopSettings.shop_name, 14, true);
  } else {
    addCenteredText('CYBER SHOP', 14, true);
  }

  if (shopSettings?.address) {
    addCenteredText(shopSettings.address, 8);
  }

  if (shopSettings?.phone_number) {
    addCenteredText(`Tel: ${shopSettings.phone_number}`, 8);
  }

  if (shopSettings?.email) {
    addCenteredText(shopSettings.email, 8);
  }

  if (shopSettings?.mpesa_till_number) {
    addCenteredText(`M-Pesa Till: ${shopSettings.mpesa_till_number}`, 8);
  }

  yPosition += 2;
  addLine();

  // Receipt header
  if (shopSettings?.receipt_header) {
    addCenteredText(shopSettings.receipt_header, 8);
    yPosition += 2;
  }

  // Receipt details
  addLeftText(`Receipt: ${sale.receipt_number}`, 8);
  addLeftText(`Date: ${format(new Date(sale.created_at), 'dd/MM/yyyy HH:mm')}`, 8);
  addLeftText(`Cashier: ${cashierName}`, 8);
  
  if (sale.customer_name) {
    addLeftText(`Customer: ${sale.customer_name}`, 8);
  }

  yPosition += 2;
  addLine();

  // Items header
  pdf.setFontSize(8);
  pdf.setFont('helvetica', 'bold');
  pdf.text('Item', 5, yPosition);
  pdf.text('Qty', 35, yPosition);
  pdf.text('Price', 50, yPosition);
  pdf.text('Total', 65, yPosition);
  yPosition += lineHeight;
  addLine();

  // Items
  pdf.setFont('helvetica', 'normal');
  sale.sale_items?.forEach(item => {
    const serviceName = item.service?.name || 'Unknown Service';
    
    // Wrap long service names
    if (serviceName.length > 15) {
      const wrapped = serviceName.substring(0, 15) + '...';
      pdf.text(wrapped, 5, yPosition);
    } else {
      pdf.text(serviceName, 5, yPosition);
    }
    
    pdf.text(item.quantity.toString(), 35, yPosition);
    pdf.text(item.unit_price.toFixed(0), 50, yPosition);
    pdf.text(item.total_price.toFixed(0), 65, yPosition);
    yPosition += lineHeight;
  });

  yPosition += 2;
  addLine();

  // Totals
  pdf.setFont('helvetica', 'normal');
  addLeftText(`Subtotal: KSh ${sale.subtotal.toFixed(2)}`);
  
  if (sale.tax_amount > 0) {
    addLeftText(`Tax: KSh ${sale.tax_amount.toFixed(2)}`);
  }
  
  if (sale.discount_amount > 0) {
    addLeftText(`Discount: KSh ${sale.discount_amount.toFixed(2)}`);
  }

  addLine();
  pdf.setFont('helvetica', 'bold');
  addLeftText(`TOTAL: KSh ${sale.total_amount.toFixed(2)}`, 10);

  yPosition += 2;
  addLine();

  // Payment info
  pdf.setFont('helvetica', 'normal');
  addLeftText(`Payment: ${sale.payment_method.toUpperCase()}`, 8);
  
  if (sale.payment_reference) {
    addLeftText(`Ref: ${sale.payment_reference}`, 8);
  }

  yPosition += 4;

  // Footer
  if (shopSettings?.receipt_footer) {
    addCenteredText(shopSettings.receipt_footer, 8);
  }

  addCenteredText('Powered by CyberPOS', 6);

  // Auto-print
  pdf.autoPrint();
  
  // Open in new window for printing
  const pdfBlob = pdf.output('blob');
  const url = URL.createObjectURL(pdfBlob);
  window.open(url, '_blank');
};