<?php
require_once 'config/config.php';
require_once 'config/database.php';
require_once 'includes/auth.php';
require_once 'includes/functions.php';

$auth->requireAdmin();

$pageTitle = 'User Management';

// Handle form submissions
if ($_SERVER['REQUEST_METHOD'] == 'POST') {
    if (!$auth->validateCSRFToken($_POST['csrf_token'] ?? '')) {
        die('CSRF token validation failed');
    }

    $action = $_POST['action'] ?? '';

    if ($action === 'add' || $action === 'edit') {
        $fullName = sanitizeInput($_POST['full_name']);
        $email = sanitizeInput($_POST['email']);
        $password = $_POST['password'] ?? '';
        $role = sanitizeInput($_POST['role']);

        if (empty($fullName) || empty($email) || !validateEmail($email)) {
            $error = 'Please fill in all required fields with valid data';
        } elseif ($action === 'add' && empty($password)) {
            $error = 'Password is required for new users';
        } elseif (!in_array($role, ['admin', 'cashier'])) {
            $error = 'Invalid role selected';
        } else {
            try {
                if ($action === 'add') {
                    // Check if email already exists
                    $stmt = $db->prepare("SELECT id FROM users WHERE email = ?");
                    $stmt->execute([$email]);
                    if ($stmt->fetch()) {
                        $error = 'Email address already exists';
                    } else {
                        $hashedPassword = password_hash($password, PASSWORD_DEFAULT);
                        $stmt = $db->prepare("
                            INSERT INTO users (email, password, full_name, role)
                            VALUES (?, ?, ?, ?)
                        ");
                        $stmt->execute([$email, $hashedPassword, $fullName, $role]);
                        $success = 'User created successfully';
                    }
                } else {
                    $userId = intval($_POST['user_id']);
                    $updateData = [$fullName, $role, $userId];
                    $sql = "UPDATE users SET full_name = ?, role = ? WHERE id = ?";
                    
                    if (!empty($password)) {
                        $hashedPassword = password_hash($password, PASSWORD_DEFAULT);
                        $sql = "UPDATE users SET full_name = ?, role = ?, password = ? WHERE id = ?";
                        $updateData = [$fullName, $role, $hashedPassword, $userId];
                    }
                    
                    $stmt = $db->prepare($sql);
                    $stmt->execute($updateData);
                    $success = 'User updated successfully';
                }
            } catch (Exception $e) {
                logError("Error saving user: " . $e->getMessage());
                $error = 'Failed to save user';
            }
        }
    } elseif ($action === 'delete') {
        $userId = intval($_POST['user_id']);
        
        // Prevent deleting current user
        if ($userId == $_SESSION['user_id']) {
            $error = 'You cannot delete your own account';
        } else {
            try {
                $stmt = $db->prepare("DELETE FROM users WHERE id = ?");
                $stmt->execute([$userId]);
                $success = 'User deleted successfully';
            } catch (Exception $e) {
                logError("Error deleting user: " . $e->getMessage());
                $error = 'Failed to delete user';
            }
        }
    }
}

// Get users
try {
    $stmt = $db->prepare("SELECT * FROM users ORDER BY created_at DESC");
    $stmt->execute();
    $users = $stmt->fetchAll();
} catch (Exception $e) {
    logError("Error fetching users: " . $e->getMessage());
    $users = [];
}

include 'includes/header.php';
?>

<div class="space-y-8">
    <div class="flex items-center justify-between">
        <div>
            <h1 class="text-3xl font-bold text-gray-900">User Management</h1>
            <p class="mt-2 text-gray-600">Manage staff accounts and permissions</p>
        </div>
        <button
            onclick="openUserModal()"
            class="bg-blue-600 text-white px-4 py-2 rounded-lg hover:bg-blue-700 flex items-center space-x-2 transition-colors"
        >
            <i class="fas fa-plus"></i>
            <span>Add User</span>
        </button>
    </div>

    <?php if (isset($error)): ?>
    <div class="bg-red-50 border border-red-200 text-red-700 px-4 py-3 rounded-lg">
        <i class="fas fa-exclamation-circle mr-2"></i>
        <?php echo htmlspecialchars($error); ?>
    </div>
    <?php endif; ?>

    <?php if (isset($success)): ?>
    <div class="bg-green-50 border border-green-200 text-green-700 px-4 py-3 rounded-lg">
        <i class="fas fa-check-circle mr-2"></i>
        <?php echo htmlspecialchars($success); ?>
    </div>
    <?php endif; ?>

    <div class="bg-white rounded-xl shadow-sm border border-gray-100">
        <div class="overflow-x-auto">
            <table class="min-w-full divide-y divide-gray-200">
                <thead class="bg-gray-50">
                    <tr>
                        <th class="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider">
                            User
                        </th>
                        <th class="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider">
                            Role
                        </th>
                        <th class="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider">
                            Created
                        </th>
                        <th class="px-6 py-3 text-right text-xs font-medium text-gray-500 uppercase tracking-wider">
                            Actions
                        </th>
                    </tr>
                </thead>
                <tbody class="bg-white divide-y divide-gray-200">
                    <?php foreach ($users as $user): ?>
                    <tr class="hover:bg-gray-50">
                        <td class="px-6 py-4 whitespace-nowrap">
                            <div class="flex items-center">
                                <div class="h-10 w-10 rounded-full bg-blue-600 flex items-center justify-center">
                                    <span class="text-sm font-medium text-white">
                                        <?php echo strtoupper(substr($user['full_name'], 0, 1)); ?>
                                    </span>
                                </div>
                                <div class="ml-4">
                                    <div class="text-sm font-medium text-gray-900"><?php echo htmlspecialchars($user['full_name']); ?></div>
                                    <div class="text-sm text-gray-500"><?php echo htmlspecialchars($user['email']); ?></div>
                                </div>
                            </div>
                        </td>
                        <td class="px-6 py-4 whitespace-nowrap">
                            <span class="inline-flex items-center px-2.5 py-0.5 rounded-full text-xs font-medium <?php echo $user['role'] === 'admin' ? 'bg-purple-100 text-purple-800' : 'bg-blue-100 text-blue-800'; ?>">
                                <?php echo ucfirst($user['role']); ?>
                            </span>
                        </td>
                        <td class="px-6 py-4 whitespace-nowrap text-sm text-gray-500">
                            <?php echo formatDate($user['created_at'], 'M d, Y'); ?>
                        </td>
                        <td class="px-6 py-4 whitespace-nowrap text-right text-sm font-medium">
                            <div class="flex items-center justify-end space-x-2">
                                <button
                                    onclick="editUser(<?php echo htmlspecialchars(json_encode($user)); ?>)"
                                    class="text-blue-600 hover:text-blue-900 p-1 rounded"
                                >
                                    <i class="fas fa-edit"></i>
                                </button>
                                <?php if ($user['id'] != $_SESSION['user_id']): ?>
                                <button
                                    onclick="deleteUser(<?php echo $user['id']; ?>, '<?php echo htmlspecialchars($user['full_name']); ?>')"
                                    class="text-red-600 hover:text-red-900 p-1 rounded"
                                >
                                    <i class="fas fa-trash"></i>
                                </button>
                                <?php endif; ?>
                            </div>
                        </td>
                    </tr>
                    <?php endforeach; ?>
                </tbody>
            </table>
        </div>

        <?php if (empty($users)): ?>
        <div class="text-center py-12">
            <i class="fas fa-users text-4xl text-gray-400 mb-4"></i>
            <p class="text-gray-500">No users found</p>
            <p class="text-sm text-gray-400">Add your first user to get started</p>
        </div>
        <?php endif; ?>
    </div>
</div>

<!-- User Modal -->
<div id="userModal" class="fixed inset-0 z-50 overflow-y-auto hidden">
    <div class="flex min-h-screen items-center justify-center p-4">
        <div class="fixed inset-0 bg-black bg-opacity-25" onclick="closeUserModal()"></div>
        
        <div class="relative bg-white rounded-xl shadow-xl max-w-md w-full p-6">
            <div class="flex items-center justify-between mb-6">
                <h3 id="userModalTitle" class="text-lg font-semibold text-gray-900">Add New User</h3>
                <button onclick="closeUserModal()" class="text-gray-400 hover:text-gray-600">
                    <i class="fas fa-times"></i>
                </button>
            </div>

            <form id="userForm" method="POST">
                <input type="hidden" name="csrf_token" value="<?php echo $auth->generateCSRFToken(); ?>">
                <input type="hidden" name="action" id="userFormAction" value="add">
                <input type="hidden" name="user_id" id="userId">

                <div class="space-y-4">
                    <div>
                        <label class="block text-sm font-medium text-gray-700 mb-1">
                            Full Name *
                        </label>
                        <input
                            type="text"
                            name="full_name"
                            id="userFullName"
                            required
                            class="w-full px-3 py-2 border border-gray-300 rounded-lg focus:ring-2 focus:ring-blue-500 focus:border-blue-500"
                            placeholder="Enter full name"
                        />
                    </div>

                    <div>
                        <label class="block text-sm font-medium text-gray-700 mb-1">
                            Email Address *
                        </label>
                        <input
                            type="email"
                            name="email"
                            id="userEmail"
                            required
                            class="w-full px-3 py-2 border border-gray-300 rounded-lg focus:ring-2 focus:ring-blue-500 focus:border-blue-500"
                            placeholder="Enter email address"
                        />
                    </div>

                    <div id="passwordField">
                        <label class="block text-sm font-medium text-gray-700 mb-1">
                            Password *
                        </label>
                        <input
                            type="password"
                            name="password"
                            id="userPassword"
                            class="w-full px-3 py-2 border border-gray-300 rounded-lg focus:ring-2 focus:ring-blue-500 focus:border-blue-500"
                            placeholder="Enter password"
                            minlength="6"
                        />
                        <p class="text-xs text-gray-500 mt-1">Leave blank to keep current password (edit mode)</p>
                    </div>

                    <div>
                        <label class="block text-sm font-medium text-gray-700 mb-1">
                            Role *
                        </label>
                        <select
                            name="role"
                            id="userRole"
                            class="w-full px-3 py-2 border border-gray-300 rounded-lg focus:ring-2 focus:ring-blue-500 focus:border-blue-500"
                        >
                            <option value="cashier">Cashier</option>
                            <option value="admin">Admin</option>
                        </select>
                    </div>
                </div>

                <div class="flex space-x-3 pt-6">
                    <button
                        type="button"
                        onclick="closeUserModal()"
                        class="flex-1 bg-gray-100 text-gray-700 py-2 px-4 rounded-lg hover:bg-gray-200 transition-colors"
                    >
                        Cancel
                    </button>
                    <button
                        type="submit"
                        class="flex-1 bg-blue-600 text-white py-2 px-4 rounded-lg hover:bg-blue-700 transition-colors"
                    >
                        <span id="userSubmitText">Create</span>
                    </button>
                </div>
            </form>
        </div>
    </div>
</div>

<script>
function openUserModal(user = null) {
    const modal = document.getElementById('userModal');
    const form = document.getElementById('userForm');
    const title = document.getElementById('userModalTitle');
    const submitText = document.getElementById('userSubmitText');
    const action = document.getElementById('userFormAction');
    const passwordField = document.getElementById('passwordField');

    if (user) {
        // Edit mode
        title.textContent = 'Edit User';
        submitText.textContent = 'Update';
        action.value = 'edit';
        document.getElementById('userId').value = user.id;
        document.getElementById('userFullName').value = user.full_name;
        document.getElementById('userEmail').value = user.email;
        document.getElementById('userEmail').disabled = true;
        document.getElementById('userPassword').required = false;
        document.getElementById('userRole').value = user.role;
        passwordField.querySelector('p').style.display = 'block';
    } else {
        // Add mode
        title.textContent = 'Add New User';
        submitText.textContent = 'Create';
        action.value = 'add';
        form.reset();
        document.getElementById('userEmail').disabled = false;
        document.getElementById('userPassword').required = true;
        passwordField.querySelector('p').style.display = 'none';
    }

    modal.classList.remove('hidden');
}

function closeUserModal() {
    document.getElementById('userModal').classList.add('hidden');
}

function editUser(user) {
    openUserModal(user);
}

function deleteUser(userId, userName) {
    if (confirm(`Are you sure you want to delete "${userName}"?`)) {
        const form = document.createElement('form');
        form.method = 'POST';
        form.innerHTML = `
            <input type="hidden" name="csrf_token" value="<?php echo $auth->generateCSRFToken(); ?>">
            <input type="hidden" name="action" value="delete">
            <input type="hidden" name="user_id" value="${userId}">
        `;
        document.body.appendChild(form);
        form.submit();
    }
}
</script>

</main>
</body>
</html>