<?php
require_once 'config/config.php';
require_once 'config/database.php';
require_once 'includes/auth.php';
require_once 'includes/functions.php';

$auth->requireLogin();

$pageTitle = 'Transaction History';

// Get filter parameters
$dateFilter = $_GET['date'] ?? 'today';
$cashierFilter = $_GET['cashier'] ?? '';
$paymentFilter = $_GET['payment'] ?? '';
$page = intval($_GET['page'] ?? 1);
$limit = 20;
$offset = ($page - 1) * $limit;

// Build WHERE clause
$whereConditions = [];
$params = [];

// Date filter
switch ($dateFilter) {
    case 'today':
        $whereConditions[] = "DATE(s.created_at) = CURDATE()";
        break;
    case 'yesterday':
        $whereConditions[] = "DATE(s.created_at) = DATE_SUB(CURDATE(), INTERVAL 1 DAY)";
        break;
    case 'week':
        $whereConditions[] = "YEARWEEK(s.created_at) = YEARWEEK(NOW())";
        break;
    case 'month':
        $whereConditions[] = "YEAR(s.created_at) = YEAR(NOW()) AND MONTH(s.created_at) = MONTH(NOW())";
        break;
}

// Cashier filter
if (!empty($cashierFilter)) {
    $whereConditions[] = "s.cashier_id = ?";
    $params[] = $cashierFilter;
}

// Payment method filter
if (!empty($paymentFilter)) {
    $whereConditions[] = "s.payment_method = ?";
    $params[] = $paymentFilter;
}

// Admin can see all, cashiers only their own
if (!$auth->isAdmin()) {
    $whereConditions[] = "s.cashier_id = ?";
    $params[] = $_SESSION['user_id'];
}

$whereClause = !empty($whereConditions) ? 'WHERE ' . implode(' AND ', $whereConditions) : '';

try {
    // Get total count
    $countSql = "
        SELECT COUNT(*) as total
        FROM sales s
        JOIN users u ON s.cashier_id = u.id
        $whereClause
    ";
    $stmt = $db->prepare($countSql);
    $stmt->execute($params);
    $totalRecords = $stmt->fetch()['total'];
    $totalPages = ceil($totalRecords / $limit);

    // Get transactions
    $sql = "
        SELECT 
            s.*,
            u.full_name as cashier_name,
            COUNT(si.id) as item_count
        FROM sales s
        JOIN users u ON s.cashier_id = u.id
        LEFT JOIN sale_items si ON s.id = si.sale_id
        $whereClause
        GROUP BY s.id
        ORDER BY s.created_at DESC
        LIMIT ? OFFSET ?
    ";
    $params[] = $limit;
    $params[] = $offset;
    $stmt = $db->prepare($sql);
    $stmt->execute($params);
    $transactions = $stmt->fetchAll();

    // Get cashiers for filter
    $stmt = $db->prepare("SELECT id, full_name FROM users WHERE role IN ('admin', 'cashier') ORDER BY full_name");
    $stmt->execute();
    $cashiers = $stmt->fetchAll();

} catch (Exception $e) {
    logError("Error fetching transactions: " . $e->getMessage());
    $transactions = [];
    $cashiers = [];
    $totalRecords = 0;
    $totalPages = 0;
}

include 'includes/header.php';
?>

<div class="space-y-8">
    <div>
        <h1 class="text-3xl font-bold text-gray-900">Transaction History</h1>
        <p class="mt-2 text-gray-600">View and search transaction records</p>
    </div>

    <!-- Filters -->
    <div class="bg-white rounded-xl shadow-sm border border-gray-100 p-6">
        <form method="GET" class="grid grid-cols-1 md:grid-cols-4 gap-4">
            <div>
                <label class="block text-sm font-medium text-gray-700 mb-2">Date Range</label>
                <select name="date" class="w-full px-3 py-2 border border-gray-300 rounded-lg focus:ring-2 focus:ring-blue-500 focus:border-blue-500">
                    <option value="today" <?php echo $dateFilter === 'today' ? 'selected' : ''; ?>>Today</option>
                    <option value="yesterday" <?php echo $dateFilter === 'yesterday' ? 'selected' : ''; ?>>Yesterday</option>
                    <option value="week" <?php echo $dateFilter === 'week' ? 'selected' : ''; ?>>This Week</option>
                    <option value="month" <?php echo $dateFilter === 'month' ? 'selected' : ''; ?>>This Month</option>
                    <option value="all" <?php echo $dateFilter === 'all' ? 'selected' : ''; ?>>All Time</option>
                </select>
            </div>

            <?php if ($auth->isAdmin()): ?>
            <div>
                <label class="block text-sm font-medium text-gray-700 mb-2">Cashier</label>
                <select name="cashier" class="w-full px-3 py-2 border border-gray-300 rounded-lg focus:ring-2 focus:ring-blue-500 focus:border-blue-500">
                    <option value="">All Cashiers</option>
                    <?php foreach ($cashiers as $cashier): ?>
                    <option value="<?php echo $cashier['id']; ?>" <?php echo $cashierFilter == $cashier['id'] ? 'selected' : ''; ?>>
                        <?php echo htmlspecialchars($cashier['full_name']); ?>
                    </option>
                    <?php endforeach; ?>
                </select>
            </div>
            <?php endif; ?>

            <div>
                <label class="block text-sm font-medium text-gray-700 mb-2">Payment Method</label>
                <select name="payment" class="w-full px-3 py-2 border border-gray-300 rounded-lg focus:ring-2 focus:ring-blue-500 focus:border-blue-500">
                    <option value="">All Methods</option>
                    <option value="cash" <?php echo $paymentFilter === 'cash' ? 'selected' : ''; ?>>Cash</option>
                    <option value="mpesa" <?php echo $paymentFilter === 'mpesa' ? 'selected' : ''; ?>>M-Pesa</option>
                    <option value="card" <?php echo $paymentFilter === 'card' ? 'selected' : ''; ?>>Card</option>
                </select>
            </div>

            <div class="flex items-end">
                <button
                    type="submit"
                    class="w-full bg-blue-600 text-white px-4 py-2 rounded-lg hover:bg-blue-700 transition-colors"
                >
                    Apply Filters
                </button>
            </div>
        </form>
    </div>

    <!-- Transactions Table -->
    <div class="bg-white rounded-xl shadow-sm border border-gray-100">
        <div class="overflow-x-auto">
            <table class="min-w-full divide-y divide-gray-200">
                <thead class="bg-gray-50">
                    <tr>
                        <th class="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider">
                            Receipt
                        </th>
                        <th class="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider">
                            Customer
                        </th>
                        <th class="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider">
                            Items
                        </th>
                        <th class="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider">
                            Payment
                        </th>
                        <th class="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider">
                            Total
                        </th>
                        <th class="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider">
                            Cashier
                        </th>
                        <th class="px-6 py-3 text-right text-xs font-medium text-gray-500 uppercase tracking-wider">
                            Actions
                        </th>
                    </tr>
                </thead>
                <tbody class="bg-white divide-y divide-gray-200">
                    <?php foreach ($transactions as $transaction): ?>
                    <tr class="hover:bg-gray-50">
                        <td class="px-6 py-4 whitespace-nowrap">
                            <div>
                                <div class="text-sm font-medium text-gray-900"><?php echo htmlspecialchars($transaction['receipt_number']); ?></div>
                                <div class="text-sm text-gray-500"><?php echo formatDate($transaction['created_at'], 'M d, Y H:i'); ?></div>
                            </div>
                        </td>
                        <td class="px-6 py-4 whitespace-nowrap text-sm text-gray-900">
                            <?php echo $transaction['customer_name'] ? htmlspecialchars($transaction['customer_name']) : 'Walk-in'; ?>
                        </td>
                        <td class="px-6 py-4 whitespace-nowrap text-sm text-gray-500">
                            <?php echo $transaction['item_count']; ?> items
                        </td>
                        <td class="px-6 py-4 whitespace-nowrap">
                            <span class="inline-flex items-center px-2.5 py-0.5 rounded-full text-xs font-medium bg-gray-100 text-gray-800 capitalize">
                                <?php echo $transaction['payment_method']; ?>
                            </span>
                        </td>
                        <td class="px-6 py-4 whitespace-nowrap text-sm font-medium text-gray-900">
                            <?php echo formatCurrency($transaction['total_amount']); ?>
                        </td>
                        <td class="px-6 py-4 whitespace-nowrap text-sm text-gray-500">
                            <?php echo htmlspecialchars($transaction['cashier_name']); ?>
                        </td>
                        <td class="px-6 py-4 whitespace-nowrap text-right text-sm font-medium">
                            <div class="flex items-center justify-end space-x-2">
                                <button
                                    onclick="viewTransaction(<?php echo $transaction['id']; ?>)"
                                    class="text-blue-600 hover:text-blue-900 p-1 rounded"
                                    title="View Details"
                                >
                                    <i class="fas fa-eye"></i>
                                </button>
                                <a
                                    href="receipts/generate.php?sale_id=<?php echo $transaction['id']; ?>"
                                    target="_blank"
                                    class="text-green-600 hover:text-green-900 p-1 rounded"
                                    title="Print Receipt"
                                >
                                    <i class="fas fa-print"></i>
                                </a>
                            </div>
                        </td>
                    </tr>
                    <?php endforeach; ?>
                </tbody>
            </table>
        </div>

        <?php if (empty($transactions)): ?>
        <div class="text-center py-12">
            <i class="fas fa-receipt text-4xl text-gray-400 mb-4"></i>
            <p class="text-gray-500">No transactions found</p>
            <p class="text-sm text-gray-400">Transactions will appear here once sales are made</p>
        </div>
        <?php endif; ?>
    </div>

    <!-- Pagination -->
    <?php if ($totalPages > 1): ?>
    <div class="flex items-center justify-between">
        <div class="text-sm text-gray-700">
            Showing <?php echo ($offset + 1); ?> to <?php echo min($offset + $limit, $totalRecords); ?> of <?php echo $totalRecords; ?> results
        </div>
        <div class="flex space-x-2">
            <?php if ($page > 1): ?>
            <a href="?<?php echo http_build_query(array_merge($_GET, ['page' => $page - 1])); ?>" 
               class="px-3 py-2 bg-white border border-gray-300 rounded-lg text-sm text-gray-700 hover:bg-gray-50">
                Previous
            </a>
            <?php endif; ?>
            
            <?php for ($i = max(1, $page - 2); $i <= min($totalPages, $page + 2); $i++): ?>
            <a href="?<?php echo http_build_query(array_merge($_GET, ['page' => $i])); ?>" 
               class="px-3 py-2 <?php echo $i === $page ? 'bg-blue-600 text-white' : 'bg-white text-gray-700 hover:bg-gray-50'; ?> border border-gray-300 rounded-lg text-sm">
                <?php echo $i; ?>
            </a>
            <?php endfor; ?>
            
            <?php if ($page < $totalPages): ?>
            <a href="?<?php echo http_build_query(array_merge($_GET, ['page' => $page + 1])); ?>" 
               class="px-3 py-2 bg-white border border-gray-300 rounded-lg text-sm text-gray-700 hover:bg-gray-50">
                Next
            </a>
            <?php endif; ?>
        </div>
    </div>
    <?php endif; ?>
</div>

<!-- Transaction Details Modal -->
<div id="transactionModal" class="fixed inset-0 z-50 overflow-y-auto hidden">
    <div class="flex min-h-screen items-center justify-center p-4">
        <div class="fixed inset-0 bg-black bg-opacity-25" onclick="closeTransactionModal()"></div>
        
        <div class="relative bg-white rounded-xl shadow-xl max-w-2xl w-full p-6">
            <div class="flex items-center justify-between mb-6">
                <h3 class="text-lg font-semibold text-gray-900">Transaction Details</h3>
                <button onclick="closeTransactionModal()" class="text-gray-400 hover:text-gray-600">
                    <i class="fas fa-times"></i>
                </button>
            </div>

            <div id="transactionDetails">
                <!-- Transaction details will be loaded here -->
            </div>
        </div>
    </div>
</div>

<script>
function viewTransaction(saleId) {
    const modal = document.getElementById('transactionModal');
    const detailsDiv = document.getElementById('transactionDetails');
    
    detailsDiv.innerHTML = '<div class="text-center py-8"><i class="fas fa-spinner fa-spin text-2xl text-gray-400"></i></div>';
    modal.classList.remove('hidden');

    fetch(`api/transaction_details.php?sale_id=${saleId}`)
        .then(response => response.json())
        .then(data => {
            if (data.success) {
                const sale = data.data;
                detailsDiv.innerHTML = `
                    <div class="space-y-6">
                        <div class="grid grid-cols-2 gap-4">
                            <div>
                                <label class="block text-sm font-medium text-gray-700">Receipt Number</label>
                                <p class="text-sm text-gray-900">${sale.receipt_number}</p>
                            </div>
                            <div>
                                <label class="block text-sm font-medium text-gray-700">Date & Time</label>
                                <p class="text-sm text-gray-900">${new Date(sale.created_at).toLocaleString()}</p>
                            </div>
                            <div>
                                <label class="block text-sm font-medium text-gray-700">Customer</label>
                                <p class="text-sm text-gray-900">${sale.customer_name || 'Walk-in'}</p>
                            </div>
                            <div>
                                <label class="block text-sm font-medium text-gray-700">Cashier</label>
                                <p class="text-sm text-gray-900">${sale.cashier_name}</p>
                            </div>
                        </div>

                        <div>
                            <label class="block text-sm font-medium text-gray-700 mb-2">Items</label>
                            <div class="border border-gray-200 rounded-lg overflow-hidden">
                                <table class="min-w-full divide-y divide-gray-200">
                                    <thead class="bg-gray-50">
                                        <tr>
                                            <th class="px-4 py-2 text-left text-xs font-medium text-gray-500">Service</th>
                                            <th class="px-4 py-2 text-center text-xs font-medium text-gray-500">Qty</th>
                                            <th class="px-4 py-2 text-right text-xs font-medium text-gray-500">Price</th>
                                            <th class="px-4 py-2 text-right text-xs font-medium text-gray-500">Total</th>
                                        </tr>
                                    </thead>
                                    <tbody class="divide-y divide-gray-200">
                                        ${sale.items.map(item => `
                                            <tr>
                                                <td class="px-4 py-2 text-sm text-gray-900">${item.service_name}</td>
                                                <td class="px-4 py-2 text-sm text-gray-900 text-center">${item.quantity}</td>
                                                <td class="px-4 py-2 text-sm text-gray-900 text-right">KSh ${parseFloat(item.unit_price).toLocaleString()}</td>
                                                <td class="px-4 py-2 text-sm text-gray-900 text-right">KSh ${parseFloat(item.total_price).toLocaleString()}</td>
                                            </tr>
                                        `).join('')}
                                    </tbody>
                                </table>
                            </div>
                        </div>

                        <div class="grid grid-cols-2 gap-4 pt-4 border-t border-gray-200">
                            <div>
                                <label class="block text-sm font-medium text-gray-700">Payment Method</label>
                                <p class="text-sm text-gray-900 capitalize">${sale.payment_method}</p>
                                ${sale.payment_reference ? `<p class="text-xs text-gray-500">Ref: ${sale.payment_reference}</p>` : ''}
                            </div>
                            <div class="text-right">
                                <div class="space-y-1">
                                    <div class="flex justify-between text-sm">
                                        <span>Subtotal:</span>
                                        <span>KSh ${parseFloat(sale.subtotal).toLocaleString()}</span>
                                    </div>
                                    ${sale.tax_amount > 0 ? `
                                    <div class="flex justify-between text-sm">
                                        <span>Tax:</span>
                                        <span>KSh ${parseFloat(sale.tax_amount).toLocaleString()}</span>
                                    </div>
                                    ` : ''}
                                    ${sale.discount_amount > 0 ? `
                                    <div class="flex justify-between text-sm text-green-600">
                                        <span>Discount:</span>
                                        <span>-KSh ${parseFloat(sale.discount_amount).toLocaleString()}</span>
                                    </div>
                                    ` : ''}
                                    <div class="flex justify-between text-lg font-bold border-t border-gray-200 pt-2">
                                        <span>Total:</span>
                                        <span>KSh ${parseFloat(sale.total_amount).toLocaleString()}</span>
                                    </div>
                                </div>
                            </div>
                        </div>

                        <div class="flex justify-end space-x-3 pt-4">
                            <a
                                href="receipts/generate.php?sale_id=${sale.id}"
                                target="_blank"
                                class="bg-green-600 text-white px-4 py-2 rounded-lg hover:bg-green-700 flex items-center space-x-2 transition-colors"
                            >
                                <i class="fas fa-print"></i>
                                <span>Print Receipt</span>
                            </a>
                        </div>
                    </div>
                `;
            } else {
                detailsDiv.innerHTML = '<div class="text-center py-8 text-red-600">Failed to load transaction details</div>';
            }
        })
        .catch(error => {
            detailsDiv.innerHTML = '<div class="text-center py-8 text-red-600">Error loading transaction details</div>';
        });
}

function closeTransactionModal() {
    document.getElementById('transactionModal').classList.add('hidden');
}
</script>

</main>
</body>
</html>