<?php
require_once 'config/config.php';
require_once 'config/database.php';
require_once 'includes/auth.php';
require_once 'includes/functions.php';

$auth->requireAdmin();

$pageTitle = 'Sales Reports';

// Get date range
$dateRange = $_GET['range'] ?? 'today';
$startDate = $_GET['start_date'] ?? date('Y-m-d');
$endDate = $_GET['end_date'] ?? date('Y-m-d');

// Calculate actual date range
switch ($dateRange) {
    case 'today':
        $start = date('Y-m-d 00:00:00');
        $end = date('Y-m-d 23:59:59');
        break;
    case 'yesterday':
        $start = date('Y-m-d 00:00:00', strtotime('-1 day'));
        $end = date('Y-m-d 23:59:59', strtotime('-1 day'));
        break;
    case 'week':
        $start = date('Y-m-d 00:00:00', strtotime('monday this week'));
        $end = date('Y-m-d 23:59:59', strtotime('sunday this week'));
        break;
    case 'month':
        $start = date('Y-m-01 00:00:00');
        $end = date('Y-m-t 23:59:59');
        break;
    case 'custom':
        $start = $startDate . ' 00:00:00';
        $end = $endDate . ' 23:59:59';
        break;
    default:
        $start = date('Y-m-d 00:00:00');
        $end = date('Y-m-d 23:59:59');
}

try {
    // Get sales summary
    $stmt = $db->prepare("
        SELECT 
            COUNT(*) as total_transactions,
            COALESCE(SUM(total_amount), 0) as total_revenue,
            COALESCE(AVG(total_amount), 0) as avg_transaction
        FROM sales 
        WHERE created_at BETWEEN ? AND ?
    ");
    $stmt->execute([$start, $end]);
    $summary = $stmt->fetch();

    // Get service breakdown
    $stmt = $db->prepare("
        SELECT 
            srv.name as service_name,
            SUM(si.quantity) as total_quantity,
            SUM(si.total_price) as total_revenue
        FROM sale_items si
        JOIN services srv ON si.service_id = srv.id
        JOIN sales s ON si.sale_id = s.id
        WHERE s.created_at BETWEEN ? AND ?
        GROUP BY srv.id, srv.name
        ORDER BY total_revenue DESC
    ");
    $stmt->execute([$start, $end]);
    $serviceBreakdown = $stmt->fetchAll();

    // Get recent transactions
    $stmt = $db->prepare("
        SELECT s.*, u.full_name as cashier_name
        FROM sales s
        JOIN users u ON s.cashier_id = u.id
        WHERE s.created_at BETWEEN ? AND ?
        ORDER BY s.created_at DESC
        LIMIT 10
    ");
    $stmt->execute([$start, $end]);
    $recentTransactions = $stmt->fetchAll();

} catch (Exception $e) {
    logError("Error fetching reports: " . $e->getMessage());
    $summary = ['total_transactions' => 0, 'total_revenue' => 0, 'avg_transaction' => 0];
    $serviceBreakdown = [];
    $recentTransactions = [];
}

include 'includes/header.php';
?>

<div class="space-y-8">
    <div class="flex items-center justify-between">
        <div>
            <h1 class="text-3xl font-bold text-gray-900">Sales Reports</h1>
            <p class="mt-2 text-gray-600">Analyze your cyber shop performance</p>
        </div>
        <button
            onclick="exportToCSV()"
            class="bg-green-600 text-white px-4 py-2 rounded-lg hover:bg-green-700 flex items-center space-x-2 transition-colors"
        >
            <i class="fas fa-download"></i>
            <span>Export CSV</span>
        </button>
    </div>

    <!-- Date Range Selector -->
    <div class="bg-white rounded-xl shadow-sm border border-gray-100 p-6">
        <form method="GET" class="flex flex-wrap items-center gap-4">
            <div class="flex items-center space-x-2">
                <i class="fas fa-calendar text-gray-400"></i>
                <span class="font-medium text-gray-700">Date Range:</span>
            </div>
            
            <select
                name="range"
                onchange="toggleCustomDates(this.value)"
                class="px-3 py-2 border border-gray-300 rounded-lg focus:ring-2 focus:ring-blue-500 focus:border-blue-500"
            >
                <option value="today" <?php echo $dateRange === 'today' ? 'selected' : ''; ?>>Today</option>
                <option value="yesterday" <?php echo $dateRange === 'yesterday' ? 'selected' : ''; ?>>Yesterday</option>
                <option value="week" <?php echo $dateRange === 'week' ? 'selected' : ''; ?>>This Week</option>
                <option value="month" <?php echo $dateRange === 'month' ? 'selected' : ''; ?>>This Month</option>
                <option value="custom" <?php echo $dateRange === 'custom' ? 'selected' : ''; ?>>Custom Range</option>
            </select>

            <div id="customDates" class="flex items-center space-x-2" style="<?php echo $dateRange !== 'custom' ? 'display: none;' : ''; ?>">
                <input
                    type="date"
                    name="start_date"
                    value="<?php echo $startDate; ?>"
                    class="px-3 py-2 border border-gray-300 rounded-lg focus:ring-2 focus:ring-blue-500 focus:border-blue-500"
                />
                <span class="text-gray-500">to</span>
                <input
                    type="date"
                    name="end_date"
                    value="<?php echo $endDate; ?>"
                    class="px-3 py-2 border border-gray-300 rounded-lg focus:ring-2 focus:ring-blue-500 focus:border-blue-500"
                />
            </div>

            <button
                type="submit"
                class="bg-blue-600 text-white px-4 py-2 rounded-lg hover:bg-blue-700 transition-colors"
            >
                Apply Filter
            </button>
        </form>
    </div>

    <!-- Summary Cards -->
    <div class="grid grid-cols-1 md:grid-cols-3 gap-6">
        <div class="bg-white rounded-xl shadow-sm p-6 border border-gray-100">
            <div class="flex items-center">
                <div class="p-2 bg-green-100 rounded-lg">
                    <i class="fas fa-chart-line text-xl text-green-600"></i>
                </div>
                <div class="ml-4">
                    <p class="text-sm font-medium text-gray-600">Total Revenue</p>
                    <p class="text-2xl font-bold text-gray-900">
                        <?php echo formatCurrency($summary['total_revenue']); ?>
                    </p>
                </div>
            </div>
        </div>

        <div class="bg-white rounded-xl shadow-sm p-6 border border-gray-100">
            <div class="flex items-center">
                <div class="p-2 bg-blue-100 rounded-lg">
                    <i class="fas fa-receipt text-xl text-blue-600"></i>
                </div>
                <div class="ml-4">
                    <p class="text-sm font-medium text-gray-600">Transactions</p>
                    <p class="text-2xl font-bold text-gray-900"><?php echo number_format($summary['total_transactions']); ?></p>
                </div>
            </div>
        </div>

        <div class="bg-white rounded-xl shadow-sm p-6 border border-gray-100">
            <div class="flex items-center">
                <div class="p-2 bg-purple-100 rounded-lg">
                    <i class="fas fa-calculator text-xl text-purple-600"></i>
                </div>
                <div class="ml-4">
                    <p class="text-sm font-medium text-gray-600">Avg. Transaction</p>
                    <p class="text-2xl font-bold text-gray-900">
                        <?php echo formatCurrency($summary['avg_transaction']); ?>
                    </p>
                </div>
            </div>
        </div>
    </div>

    <!-- Service Breakdown and Recent Transactions -->
    <div class="grid grid-cols-1 lg:grid-cols-2 gap-8">
        <!-- Service Breakdown -->
        <div class="bg-white rounded-xl shadow-sm border border-gray-100">
            <div class="p-6 border-b border-gray-100">
                <h3 class="text-lg font-semibold text-gray-900">Service Breakdown</h3>
            </div>
            <div class="p-6">
                <?php if (!empty($serviceBreakdown)): ?>
                <div class="space-y-4">
                    <?php foreach ($serviceBreakdown as $service): ?>
                    <div class="flex items-center justify-between">
                        <div>
                            <p class="font-medium text-gray-900"><?php echo htmlspecialchars($service['service_name']); ?></p>
                            <p class="text-sm text-gray-500"><?php echo $service['total_quantity']; ?> units sold</p>
                        </div>
                        <p class="font-semibold text-gray-900">
                            <?php echo formatCurrency($service['total_revenue']); ?>
                        </p>
                    </div>
                    <?php endforeach; ?>
                </div>
                <?php else: ?>
                <p class="text-gray-500 text-center py-8">No data available</p>
                <?php endif; ?>
            </div>
        </div>

        <!-- Recent Transactions -->
        <div class="bg-white rounded-xl shadow-sm border border-gray-100">
            <div class="p-6 border-b border-gray-100">
                <h3 class="text-lg font-semibold text-gray-900">Recent Transactions</h3>
            </div>
            <div class="p-6">
                <?php if (!empty($recentTransactions)): ?>
                <div class="space-y-4 max-h-96 overflow-y-auto">
                    <?php foreach ($recentTransactions as $transaction): ?>
                    <div class="flex items-center justify-between p-3 bg-gray-50 rounded-lg">
                        <div>
                            <p class="font-medium text-gray-900"><?php echo htmlspecialchars($transaction['receipt_number']); ?></p>
                            <p class="text-sm text-gray-500">
                                <?php echo formatDate($transaction['created_at'], 'M d, H:i'); ?> • <?php echo htmlspecialchars($transaction['cashier_name']); ?>
                            </p>
                        </div>
                        <div class="text-right">
                            <p class="font-semibold text-gray-900">
                                <?php echo formatCurrency($transaction['total_amount']); ?>
                            </p>
                            <p class="text-sm text-gray-500 capitalize"><?php echo $transaction['payment_method']; ?></p>
                        </div>
                    </div>
                    <?php endforeach; ?>
                </div>
                <?php else: ?>
                <p class="text-gray-500 text-center py-8">No transactions found</p>
                <?php endif; ?>
            </div>
        </div>
    </div>
</div>

<script>
function toggleCustomDates(range) {
    const customDates = document.getElementById('customDates');
    if (range === 'custom') {
        customDates.style.display = 'flex';
    } else {
        customDates.style.display = 'none';
    }
}

function exportToCSV() {
    window.location.href = `api/export.php?range=<?php echo $dateRange; ?>&start_date=<?php echo $startDate; ?>&end_date=<?php echo $endDate; ?>`;
}
</script>

</main>
</body>
</html>