<?php
require_once '../config/config.php';
require_once '../config/database.php';
require_once '../includes/auth.php';
require_once '../includes/functions.php';

$auth->requireLogin();

$saleId = intval($_GET['sale_id'] ?? 0);

if (!$saleId) {
    die('Invalid sale ID');
}

try {
    // Get sale details
    $stmt = $db->prepare("
        SELECT s.*, u.full_name as cashier_name
        FROM sales s
        JOIN users u ON s.cashier_id = u.id
        WHERE s.id = ?
    ");
    $stmt->execute([$saleId]);
    $sale = $stmt->fetch();

    if (!$sale) {
        die('Sale not found');
    }

    // Get sale items
    $stmt = $db->prepare("
        SELECT si.*, srv.name as service_name
        FROM sale_items si
        JOIN services srv ON si.service_id = srv.id
        WHERE si.sale_id = ?
        ORDER BY srv.name
    ");
    $stmt->execute([$saleId]);
    $saleItems = $stmt->fetchAll();

    $shopSettings = getShopSettings($db);

} catch (Exception $e) {
    logError("Error generating receipt: " . $e->getMessage());
    die('Error generating receipt');
}
?>
<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Receipt - <?php echo htmlspecialchars($sale['receipt_number']); ?></title>
    <style>
        @media print {
            body { margin: 0; }
            .no-print { display: none; }
        }
        
        .receipt {
            width: 80mm;
            margin: 0 auto;
            font-family: 'Courier New', monospace;
            font-size: 12px;
            line-height: 1.4;
        }
        
        .center { text-align: center; }
        .bold { font-weight: bold; }
        .line { border-bottom: 1px dashed #000; margin: 5px 0; }
        
        table { width: 100%; border-collapse: collapse; }
        th, td { padding: 2px; text-align: left; }
        .right { text-align: right; }
    </style>
</head>
<body>
    <div class="receipt">
        <!-- Shop Header -->
        <div class="center bold" style="font-size: 14px;">
            <?php echo htmlspecialchars($shopSettings['shop_name'] ?? 'CYBER SHOP'); ?>
        </div>
        
        <?php if ($shopSettings['address']): ?>
        <div class="center"><?php echo nl2br(htmlspecialchars($shopSettings['address'])); ?></div>
        <?php endif; ?>
        
        <?php if ($shopSettings['phone_number']): ?>
        <div class="center">Tel: <?php echo htmlspecialchars($shopSettings['phone_number']); ?></div>
        <?php endif; ?>
        
        <?php if ($shopSettings['email']): ?>
        <div class="center"><?php echo htmlspecialchars($shopSettings['email']); ?></div>
        <?php endif; ?>
        
        <?php if ($shopSettings['mpesa_till_number']): ?>
        <div class="center">M-Pesa Till: <?php echo htmlspecialchars($shopSettings['mpesa_till_number']); ?></div>
        <?php endif; ?>

        <div class="line"></div>

        <?php if ($shopSettings['receipt_header']): ?>
        <div class="center"><?php echo nl2br(htmlspecialchars($shopSettings['receipt_header'])); ?></div>
        <div class="line"></div>
        <?php endif; ?>

        <!-- Receipt Details -->
        <div>Receipt: <?php echo htmlspecialchars($sale['receipt_number']); ?></div>
        <div>Date: <?php echo formatDate($sale['created_at'], 'd/m/Y H:i'); ?></div>
        <div>Cashier: <?php echo htmlspecialchars($sale['cashier_name']); ?></div>
        
        <?php if ($sale['customer_name']): ?>
        <div>Customer: <?php echo htmlspecialchars($sale['customer_name']); ?></div>
        <?php endif; ?>

        <div class="line"></div>

        <!-- Items -->
        <table>
            <thead>
                <tr>
                    <th>Item</th>
                    <th class="center">Qty</th>
                    <th class="right">Price</th>
                    <th class="right">Total</th>
                </tr>
            </thead>
            <tbody>
                <?php foreach ($saleItems as $item): ?>
                <tr>
                    <td><?php echo htmlspecialchars($item['service_name']); ?></td>
                    <td class="center"><?php echo $item['quantity']; ?></td>
                    <td class="right"><?php echo number_format($item['unit_price'], 0); ?></td>
                    <td class="right"><?php echo number_format($item['total_price'], 0); ?></td>
                </tr>
                <?php endforeach; ?>
            </tbody>
        </table>

        <div class="line"></div>

        <!-- Totals -->
        <table>
            <tr>
                <td>Subtotal:</td>
                <td class="right">KSh <?php echo number_format($sale['subtotal'], 2); ?></td>
            </tr>
            
            <?php if ($sale['tax_amount'] > 0): ?>
            <tr>
                <td>Tax:</td>
                <td class="right">KSh <?php echo number_format($sale['tax_amount'], 2); ?></td>
            </tr>
            <?php endif; ?>
            
            <?php if ($sale['discount_amount'] > 0): ?>
            <tr>
                <td>Discount:</td>
                <td class="right">KSh <?php echo number_format($sale['discount_amount'], 2); ?></td>
            </tr>
            <?php endif; ?>
            
            <tr class="bold">
                <td>TOTAL:</td>
                <td class="right">KSh <?php echo number_format($sale['total_amount'], 2); ?></td>
            </tr>
        </table>

        <div class="line"></div>

        <!-- Payment Info -->
        <div>Payment: <?php echo strtoupper($sale['payment_method']); ?></div>
        
        <?php if ($sale['payment_reference']): ?>
        <div>Ref: <?php echo htmlspecialchars($sale['payment_reference']); ?></div>
        <?php endif; ?>

        <div class="line"></div>

        <!-- Footer -->
        <?php if ($shopSettings['receipt_footer']): ?>
        <div class="center"><?php echo nl2br(htmlspecialchars($shopSettings['receipt_footer'])); ?></div>
        <?php endif; ?>
        
        <div class="center" style="font-size: 10px; margin-top: 10px;">
            Powered by CyberPOS
        </div>
    </div>

    <div class="no-print" style="text-align: center; margin-top: 20px;">
        <button onclick="window.print()" class="bg-blue-600 text-white px-4 py-2 rounded">
            <i class="fas fa-print mr-2"></i> Print Receipt
        </button>
        <button onclick="window.close()" class="bg-gray-600 text-white px-4 py-2 rounded ml-2">
            Close
        </button>
    </div>

    <script>
        // Auto-print receipt
        <?php if (AUTO_PRINT_RECEIPT): ?>
        window.onload = function() {
            window.print();
        };
        <?php endif; ?>
    </script>
</body>
</html>