<?php
require_once 'config/config.php';
require_once 'config/database.php';
require_once 'includes/auth.php';
require_once 'includes/functions.php';

$auth->requireLogin();

$pageTitle = 'My Profile';

// Handle form submission
if ($_SERVER['REQUEST_METHOD'] == 'POST') {
    if (!$auth->validateCSRFToken($_POST['csrf_token'] ?? '')) {
        die('CSRF token validation failed');
    }

    $fullName = sanitizeInput($_POST['full_name']);
    $currentPassword = $_POST['current_password'] ?? '';
    $newPassword = $_POST['new_password'] ?? '';
    $confirmPassword = $_POST['confirm_password'] ?? '';

    if (empty($fullName)) {
        $error = 'Full name is required';
    } elseif (!empty($newPassword)) {
        if (empty($currentPassword)) {
            $error = 'Current password is required to change password';
        } elseif ($newPassword !== $confirmPassword) {
            $error = 'New passwords do not match';
        } elseif (strlen($newPassword) < 6) {
            $error = 'New password must be at least 6 characters';
        } else {
            // Verify current password
            $currentUser = $auth->getCurrentUser();
            if (!password_verify($currentPassword, $currentUser['password'])) {
                $error = 'Current password is incorrect';
            } else {
                // Update with new password
                try {
                    $hashedPassword = password_hash($newPassword, PASSWORD_DEFAULT);
                    $stmt = $db->prepare("UPDATE users SET full_name = ?, password = ? WHERE id = ?");
                    $stmt->execute([$fullName, $hashedPassword, $_SESSION['user_id']]);
                    $_SESSION['user_name'] = $fullName;
                    $success = 'Profile updated successfully';
                } catch (Exception $e) {
                    logError("Error updating profile: " . $e->getMessage());
                    $error = 'Failed to update profile';
                }
            }
        }
    } else {
        // Update only name
        try {
            $stmt = $db->prepare("UPDATE users SET full_name = ? WHERE id = ?");
            $stmt->execute([$fullName, $_SESSION['user_id']]);
            $_SESSION['user_name'] = $fullName;
            $success = 'Profile updated successfully';
        } catch (Exception $e) {
            logError("Error updating profile: " . $e->getMessage());
            $error = 'Failed to update profile';
        }
    }
}

$currentUser = $auth->getCurrentUser();

include 'includes/header.php';
?>

<div class="space-y-8">
    <div>
        <h1 class="text-3xl font-bold text-gray-900">My Profile</h1>
        <p class="mt-2 text-gray-600">Manage your account settings</p>
    </div>

    <?php if (isset($error)): ?>
    <div class="bg-red-50 border border-red-200 text-red-700 px-4 py-3 rounded-lg">
        <i class="fas fa-exclamation-circle mr-2"></i>
        <?php echo htmlspecialchars($error); ?>
    </div>
    <?php endif; ?>

    <?php if (isset($success)): ?>
    <div class="bg-green-50 border border-green-200 text-green-700 px-4 py-3 rounded-lg">
        <i class="fas fa-check-circle mr-2"></i>
        <?php echo htmlspecialchars($success); ?>
    </div>
    <?php endif; ?>

    <div class="grid grid-cols-1 lg:grid-cols-3 gap-8">
        <!-- Profile Form -->
        <div class="lg:col-span-2">
            <div class="bg-white rounded-xl shadow-sm border border-gray-100 p-6">
                <h2 class="text-lg font-semibold text-gray-900 mb-6">Profile Information</h2>
                
                <form method="POST" class="space-y-6">
                    <input type="hidden" name="csrf_token" value="<?php echo $auth->generateCSRFToken(); ?>">

                    <div>
                        <label class="block text-sm font-medium text-gray-700 mb-2">
                            Full Name *
                        </label>
                        <input
                            type="text"
                            name="full_name"
                            required
                            value="<?php echo htmlspecialchars($currentUser['full_name']); ?>"
                            class="w-full px-3 py-2 border border-gray-300 rounded-lg focus:ring-2 focus:ring-blue-500 focus:border-blue-500"
                        />
                    </div>

                    <div>
                        <label class="block text-sm font-medium text-gray-700 mb-2">
                            Email Address
                        </label>
                        <input
                            type="email"
                            value="<?php echo htmlspecialchars($currentUser['email']); ?>"
                            disabled
                            class="w-full px-3 py-2 border border-gray-300 rounded-lg bg-gray-50 text-gray-500"
                        />
                        <p class="text-xs text-gray-500 mt-1">Email cannot be changed</p>
                    </div>

                    <div>
                        <label class="block text-sm font-medium text-gray-700 mb-2">
                            Role
                        </label>
                        <input
                            type="text"
                            value="<?php echo ucfirst($currentUser['role']); ?>"
                            disabled
                            class="w-full px-3 py-2 border border-gray-300 rounded-lg bg-gray-50 text-gray-500"
                        />
                    </div>

                    <div class="border-t border-gray-200 pt-6">
                        <h3 class="text-md font-medium text-gray-900 mb-4">Change Password</h3>
                        <p class="text-sm text-gray-600 mb-4">Leave password fields empty if you don't want to change your password</p>
                        
                        <div class="space-y-4">
                            <div>
                                <label class="block text-sm font-medium text-gray-700 mb-2">
                                    Current Password
                                </label>
                                <input
                                    type="password"
                                    name="current_password"
                                    class="w-full px-3 py-2 border border-gray-300 rounded-lg focus:ring-2 focus:ring-blue-500 focus:border-blue-500"
                                />
                            </div>

                            <div>
                                <label class="block text-sm font-medium text-gray-700 mb-2">
                                    New Password
                                </label>
                                <input
                                    type="password"
                                    name="new_password"
                                    minlength="6"
                                    class="w-full px-3 py-2 border border-gray-300 rounded-lg focus:ring-2 focus:ring-blue-500 focus:border-blue-500"
                                />
                            </div>

                            <div>
                                <label class="block text-sm font-medium text-gray-700 mb-2">
                                    Confirm New Password
                                </label>
                                <input
                                    type="password"
                                    name="confirm_password"
                                    minlength="6"
                                    class="w-full px-3 py-2 border border-gray-300 rounded-lg focus:ring-2 focus:ring-blue-500 focus:border-blue-500"
                                />
                            </div>
                        </div>
                    </div>

                    <div class="flex justify-end">
                        <button
                            type="submit"
                            class="bg-blue-600 text-white px-6 py-2 rounded-lg hover:bg-blue-700 flex items-center space-x-2 transition-colors"
                        >
                            <i class="fas fa-save"></i>
                            <span>Update Profile</span>
                        </button>
                    </div>
                </form>
            </div>
        </div>

        <!-- Profile Summary -->
        <div class="space-y-6">
            <div class="bg-white rounded-xl shadow-sm border border-gray-100 p-6">
                <div class="text-center">
                    <div class="h-20 w-20 rounded-full bg-blue-600 flex items-center justify-center mx-auto mb-4">
                        <span class="text-2xl font-bold text-white">
                            <?php echo strtoupper(substr($currentUser['full_name'], 0, 1)); ?>
                        </span>
                    </div>
                    <h3 class="text-lg font-semibold text-gray-900"><?php echo htmlspecialchars($currentUser['full_name']); ?></h3>
                    <p class="text-sm text-gray-600"><?php echo htmlspecialchars($currentUser['email']); ?></p>
                    <span class="inline-flex items-center px-2.5 py-0.5 rounded-full text-xs font-medium mt-2 <?php echo $currentUser['role'] === 'admin' ? 'bg-purple-100 text-purple-800' : 'bg-blue-100 text-blue-800'; ?>">
                        <?php echo ucfirst($currentUser['role']); ?>
                    </span>
                </div>
            </div>

            <div class="bg-white rounded-xl shadow-sm border border-gray-100 p-6">
                <h3 class="text-lg font-semibold text-gray-900 mb-4">Account Details</h3>
                <div class="space-y-3 text-sm">
                    <div class="flex justify-between">
                        <span class="text-gray-600">Member since:</span>
                        <span class="font-medium"><?php echo formatDate($currentUser['created_at'], 'M d, Y'); ?></span>
                    </div>
                    <div class="flex justify-between">
                        <span class="text-gray-600">Last updated:</span>
                        <span class="font-medium"><?php echo formatDate($currentUser['updated_at'], 'M d, Y'); ?></span>
                    </div>
                    <div class="flex justify-between">
                        <span class="text-gray-600">Status:</span>
                        <span class="font-medium text-green-600">Active</span>
                    </div>
                </div>
            </div>
        </div>
    </div>
</div>

</main>
</body>
</html>