<?php
require_once 'config/config.php';
require_once 'config/database.php';
require_once 'includes/auth.php';
require_once 'includes/functions.php';

$auth->requireLogin();

$pageTitle = 'Point of Sale';

// Get services and categories
try {
    $stmt = $db->prepare("
        SELECT s.*, c.name as category_name 
        FROM services s 
        LEFT JOIN categories c ON s.category_id = c.id 
        WHERE s.is_active = 1 
        ORDER BY s.name
    ");
    $stmt->execute();
    $services = $stmt->fetchAll();

    $stmt = $db->prepare("SELECT * FROM categories ORDER BY name");
    $stmt->execute();
    $categories = $stmt->fetchAll();

    $shopSettings = getShopSettings($db);
} catch (Exception $e) {
    logError("Error fetching POS data: " . $e->getMessage());
    $services = [];
    $categories = [];
    $shopSettings = null;
}

include 'includes/header.php';
?>

<div class="grid grid-cols-1 lg:grid-cols-3 gap-8 h-[calc(100vh-8rem)]">
    <!-- Services List -->
    <div class="lg:col-span-2 bg-white rounded-xl shadow-sm border border-gray-100">
        <div class="p-6 border-b border-gray-100">
            <h2 class="text-xl font-semibold text-gray-900">Services & Products</h2>
            <div class="mt-4">
                <input
                    type="text"
                    id="searchServices"
                    placeholder="Search services..."
                    class="w-full px-4 py-2 border border-gray-300 rounded-lg focus:ring-2 focus:ring-blue-500 focus:border-blue-500"
                />
            </div>
        </div>
        <div class="p-6 overflow-y-auto max-h-[calc(100vh-16rem)]">
            <div class="grid grid-cols-1 sm:grid-cols-2 xl:grid-cols-3 gap-4" id="servicesGrid">
                <?php foreach ($services as $service): ?>
                <div class="service-card border border-gray-200 rounded-lg p-4 hover:shadow-md transition-shadow cursor-pointer"
                     data-service='<?php echo json_encode($service); ?>'>
                    <h3 class="font-medium text-gray-900"><?php echo htmlspecialchars($service['name']); ?></h3>
                    <?php if ($service['description']): ?>
                    <p class="text-sm text-gray-600 mt-1"><?php echo htmlspecialchars($service['description']); ?></p>
                    <?php endif; ?>
                    <div class="flex items-center justify-between mt-3">
                        <span class="text-lg font-bold text-blue-600">
                            <?php echo formatCurrency($service['unit_price']); ?>
                        </span>
                        <?php if ($service['category_name']): ?>
                        <span class="text-xs bg-gray-100 text-gray-600 px-2 py-1 rounded">
                            <?php echo htmlspecialchars($service['category_name']); ?>
                        </span>
                        <?php endif; ?>
                    </div>
                </div>
                <?php endforeach; ?>
            </div>
        </div>
    </div>

    <!-- Cart and Checkout -->
    <div class="bg-white rounded-xl shadow-sm border border-gray-100">
        <div class="p-6 border-b border-gray-100">
            <div class="flex items-center space-x-2">
                <i class="fas fa-shopping-cart text-blue-600"></i>
                <h2 class="text-xl font-semibold text-gray-900">Cart</h2>
                <span id="cartCount" class="bg-blue-100 text-blue-600 text-sm px-2 py-1 rounded-full">0</span>
            </div>
        </div>

        <div class="p-6 space-y-6 overflow-y-auto max-h-[calc(100vh-20rem)]">
            <!-- Customer Info -->
            <div>
                <label class="block text-sm font-medium text-gray-700 mb-2">
                    Customer Name (Optional)
                </label>
                <input
                    type="text"
                    id="customerName"
                    class="w-full px-3 py-2 border border-gray-300 rounded-lg focus:ring-2 focus:ring-blue-500 focus:border-blue-500"
                    placeholder="Enter customer name"
                />
            </div>

            <!-- Cart Items -->
            <div id="cartItems" class="space-y-3">
                <!-- Cart items will be populated by JavaScript -->
            </div>

            <!-- Cart Summary -->
            <div id="cartSummary" class="hidden">
                <!-- Discount -->
                <div>
                    <label class="block text-sm font-medium text-gray-700 mb-2">
                        Discount (KSh)
                    </label>
                    <input
                        type="number"
                        id="discount"
                        value="0"
                        min="0"
                        class="w-full px-3 py-2 border border-gray-300 rounded-lg focus:ring-2 focus:ring-blue-500 focus:border-blue-500"
                        placeholder="0"
                    />
                </div>

                <!-- Payment Method -->
                <div>
                    <label class="block text-sm font-medium text-gray-700 mb-2">
                        Payment Method
                    </label>
                    <select
                        id="paymentMethod"
                        class="w-full px-3 py-2 border border-gray-300 rounded-lg focus:ring-2 focus:ring-blue-500 focus:border-blue-500"
                    >
                        <option value="cash">Cash</option>
                        <option value="mpesa">M-Pesa</option>
                        <option value="card">Card</option>
                    </select>
                </div>

                <div id="paymentReferenceDiv" class="hidden">
                    <label class="block text-sm font-medium text-gray-700 mb-2">
                        Payment Reference
                    </label>
                    <input
                        type="text"
                        id="paymentReference"
                        class="w-full px-3 py-2 border border-gray-300 rounded-lg focus:ring-2 focus:ring-blue-500 focus:border-blue-500"
                        placeholder="M-Pesa code or Card reference"
                    />
                </div>

                <!-- Totals -->
                <div class="space-y-2 pt-4 border-t border-gray-200">
                    <div class="flex justify-between text-sm">
                        <span>Subtotal:</span>
                        <span id="subtotalAmount">KSh 0</span>
                    </div>
                    <?php if ($shopSettings && $shopSettings['tax_rate'] > 0): ?>
                    <div class="flex justify-between text-sm">
                        <span>Tax (<?php echo $shopSettings['tax_rate']; ?>%):</span>
                        <span id="taxAmount">KSh 0</span>
                    </div>
                    <?php endif; ?>
                    <div class="flex justify-between text-sm text-green-600" id="discountRow" style="display: none;">
                        <span>Discount:</span>
                        <span id="discountAmount">-KSh 0</span>
                    </div>
                    <div class="flex justify-between text-lg font-bold border-t border-gray-200 pt-2">
                        <span>Total:</span>
                        <span id="totalAmount">KSh 0</span>
                    </div>
                </div>

                <!-- Action Buttons -->
                <div class="space-y-3">
                    <button
                        id="completeSale"
                        class="w-full bg-blue-600 text-white py-3 px-4 rounded-lg hover:bg-blue-700 flex items-center justify-center space-x-2 transition-colors"
                    >
                        <i class="fas fa-receipt"></i>
                        <span>Complete Sale</span>
                    </button>
                    <button
                        id="clearCart"
                        class="w-full bg-gray-100 text-gray-700 py-2 px-4 rounded-lg hover:bg-gray-200 transition-colors"
                    >
                        Clear Cart
                    </button>
                </div>
            </div>

            <div id="emptyCart" class="text-center py-12">
                <i class="fas fa-calculator text-4xl text-gray-400 mb-4"></i>
                <p class="text-gray-500">Cart is empty</p>
                <p class="text-sm text-gray-400">Add services to start a transaction</p>
            </div>
        </div>
    </div>
</div>

<script>
// POS JavaScript functionality
let cart = [];
const taxRate = <?php echo $shopSettings ? $shopSettings['tax_rate'] : 0; ?>;

// Add service to cart
function addToCart(service) {
    const existingItem = cart.find(item => item.id === service.id);
    if (existingItem) {
        existingItem.quantity += 1;
    } else {
        cart.push({
            id: service.id,
            name: service.name,
            unit_price: parseFloat(service.unit_price),
            quantity: 1
        });
    }
    updateCartDisplay();
}

// Update cart display
function updateCartDisplay() {
    const cartItemsDiv = document.getElementById('cartItems');
    const cartSummary = document.getElementById('cartSummary');
    const emptyCart = document.getElementById('emptyCart');
    const cartCount = document.getElementById('cartCount');

    cartCount.textContent = cart.length;

    if (cart.length === 0) {
        cartSummary.classList.add('hidden');
        emptyCart.classList.remove('hidden');
        cartItemsDiv.innerHTML = '';
        return;
    }

    cartSummary.classList.remove('hidden');
    emptyCart.classList.add('hidden');

    // Render cart items
    cartItemsDiv.innerHTML = cart.map(item => `
        <div class="flex items-center justify-between p-3 bg-gray-50 rounded-lg">
            <div class="flex-1">
                <h4 class="font-medium text-gray-900">${item.name}</h4>
                <p class="text-sm text-gray-600">KSh ${item.unit_price.toLocaleString()}</p>
            </div>
            <div class="flex items-center space-x-2">
                <button onclick="updateQuantity('${item.id}', ${item.quantity - 1})" class="p-1 rounded-full hover:bg-gray-200">
                    <i class="fas fa-minus text-sm"></i>
                </button>
                <span class="w-8 text-center font-medium">${item.quantity}</span>
                <button onclick="updateQuantity('${item.id}', ${item.quantity + 1})" class="p-1 rounded-full hover:bg-gray-200">
                    <i class="fas fa-plus text-sm"></i>
                </button>
                <button onclick="removeFromCart('${item.id}')" class="p-1 rounded-full hover:bg-red-100 text-red-600 ml-2">
                    <i class="fas fa-trash text-sm"></i>
                </button>
            </div>
        </div>
    `).join('');

    updateTotals();
}

// Update item quantity
function updateQuantity(serviceId, quantity) {
    if (quantity <= 0) {
        removeFromCart(serviceId);
        return;
    }
    
    const item = cart.find(item => item.id === serviceId);
    if (item) {
        item.quantity = quantity;
        updateCartDisplay();
    }
}

// Remove item from cart
function removeFromCart(serviceId) {
    cart = cart.filter(item => item.id !== serviceId);
    updateCartDisplay();
}

// Clear entire cart
function clearCart() {
    cart = [];
    document.getElementById('customerName').value = '';
    document.getElementById('discount').value = '0';
    document.getElementById('paymentReference').value = '';
    updateCartDisplay();
}

// Update totals
function updateTotals() {
    const subtotal = cart.reduce((sum, item) => sum + (item.unit_price * item.quantity), 0);
    const discount = parseFloat(document.getElementById('discount').value) || 0;
    const taxAmount = subtotal * (taxRate / 100);
    const total = subtotal + taxAmount - discount;

    document.getElementById('subtotalAmount').textContent = `KSh ${subtotal.toLocaleString()}`;
    document.getElementById('taxAmount').textContent = `KSh ${taxAmount.toLocaleString()}`;
    document.getElementById('discountAmount').textContent = `-KSh ${discount.toLocaleString()}`;
    document.getElementById('totalAmount').textContent = `KSh ${total.toLocaleString()}`;

    // Show/hide discount row
    const discountRow = document.getElementById('discountRow');
    if (discount > 0) {
        discountRow.style.display = 'flex';
    } else {
        discountRow.style.display = 'none';
    }
}

// Complete sale
async function completeSale() {
    if (cart.length === 0) {
        alert('Cart is empty');
        return;
    }

    const customerName = document.getElementById('customerName').value;
    const paymentMethod = document.getElementById('paymentMethod').value;
    const paymentReference = document.getElementById('paymentReference').value;
    const discount = parseFloat(document.getElementById('discount').value) || 0;

    const saleData = {
        customer_name: customerName,
        payment_method: paymentMethod,
        payment_reference: paymentReference,
        discount_amount: discount,
        cart_items: cart
    };

    try {
        const response = await fetch('api/sales.php', {
            method: 'POST',
            headers: {
                'Content-Type': 'application/json',
            },
            body: JSON.stringify(saleData)
        });

        const result = await response.json();

        if (result.success) {
            alert('Sale completed successfully!');
            // Open receipt in new window
            window.open(`receipts/generate.php?sale_id=${result.sale_id}`, '_blank');
            clearCart();
        } else {
            alert('Error: ' + result.message);
        }
    } catch (error) {
        alert('Error processing sale: ' + error.message);
    }
}

// Event listeners
document.addEventListener('DOMContentLoaded', function() {
    // Service card clicks
    document.querySelectorAll('.service-card').forEach(card => {
        card.addEventListener('click', function() {
            const service = JSON.parse(this.dataset.service);
            addToCart(service);
        });
    });

    // Search functionality
    document.getElementById('searchServices').addEventListener('input', function() {
        const searchTerm = this.value.toLowerCase();
        document.querySelectorAll('.service-card').forEach(card => {
            const service = JSON.parse(card.dataset.service);
            const matches = service.name.toLowerCase().includes(searchTerm) ||
                           (service.description && service.description.toLowerCase().includes(searchTerm));
            card.style.display = matches ? 'block' : 'none';
        });
    });

    // Payment method change
    document.getElementById('paymentMethod').addEventListener('change', function() {
        const paymentReferenceDiv = document.getElementById('paymentReferenceDiv');
        if (this.value !== 'cash') {
            paymentReferenceDiv.classList.remove('hidden');
        } else {
            paymentReferenceDiv.classList.add('hidden');
        }
    });

    // Discount input
    document.getElementById('discount').addEventListener('input', updateTotals);

    // Complete sale button
    document.getElementById('completeSale').addEventListener('click', completeSale);

    // Clear cart button
    document.getElementById('clearCart').addEventListener('click', clearCart);
});
</script>

</main>
</body>
</html>