<?php
require_once 'config/config.php';
require_once 'config/database.php';
require_once 'includes/auth.php';
require_once 'includes/functions.php';

$auth->requireAdmin();

$pageTitle = 'Backup & Restore';

$backupDir = 'backups/';
if (!file_exists($backupDir)) {
    mkdir($backupDir, 0755, true);
}

// Handle backup creation
if ($_SERVER['REQUEST_METHOD'] == 'POST' && isset($_POST['create_backup'])) {
    if (!$auth->validateCSRFToken($_POST['csrf_token'] ?? '')) {
        die('CSRF token validation failed');
    }

    try {
        $filename = 'backup_' . date('Y-m-d_H-i-s') . '.sql';
        $filepath = $backupDir . $filename;

        // Get all tables
        $tables = [];
        $stmt = $db->query("SHOW TABLES");
        while ($row = $stmt->fetch(PDO::FETCH_NUM)) {
            $tables[] = $row[0];
        }

        $backup = "-- Cyber Shop POS Backup\n";
        $backup .= "-- Created: " . date('Y-m-d H:i:s') . "\n\n";
        $backup .= "SET FOREIGN_KEY_CHECKS = 0;\n\n";

        foreach ($tables as $table) {
            // Get table structure
            $stmt = $db->query("SHOW CREATE TABLE `$table`");
            $row = $stmt->fetch(PDO::FETCH_NUM);
            $backup .= "DROP TABLE IF EXISTS `$table`;\n";
            $backup .= $row[1] . ";\n\n";

            // Get table data
            $stmt = $db->query("SELECT * FROM `$table`");
            $rows = $stmt->fetchAll(PDO::FETCH_ASSOC);

            if (!empty($rows)) {
                $columns = array_keys($rows[0]);
                $backup .= "INSERT INTO `$table` (`" . implode('`, `', $columns) . "`) VALUES\n";

                $values = [];
                foreach ($rows as $row) {
                    $rowValues = [];
                    foreach ($row as $value) {
                        if ($value === null) {
                            $rowValues[] = 'NULL';
                        } else {
                            $rowValues[] = "'" . addslashes($value) . "'";
                        }
                    }
                    $values[] = "(" . implode(', ', $rowValues) . ")";
                }
                $backup .= implode(",\n", $values) . ";\n\n";
            }
        }

        $backup .= "SET FOREIGN_KEY_CHECKS = 1;\n";

        if (file_put_contents($filepath, $backup)) {
            $success = "Backup created successfully: $filename";
        } else {
            $error = "Failed to create backup file";
        }
    } catch (Exception $e) {
        logError("Backup error: " . $e->getMessage());
        $error = "Failed to create backup";
    }
}

// Handle backup deletion
if ($_SERVER['REQUEST_METHOD'] == 'POST' && isset($_POST['delete_backup'])) {
    if (!$auth->validateCSRFToken($_POST['csrf_token'] ?? '')) {
        die('CSRF token validation failed');
    }

    $filename = $_POST['filename'];
    $filepath = $backupDir . basename($filename);

    if (file_exists($filepath) && unlink($filepath)) {
        $success = "Backup deleted successfully";
    } else {
        $error = "Failed to delete backup";
    }
}

// Get existing backups
$backups = [];
if (is_dir($backupDir)) {
    $files = scandir($backupDir);
    foreach ($files as $file) {
        if (pathinfo($file, PATHINFO_EXTENSION) === 'sql') {
            $filepath = $backupDir . $file;
            $backups[] = [
                'filename' => $file,
                'size' => filesize($filepath),
                'created' => filemtime($filepath)
            ];
        }
    }
    // Sort by creation date (newest first)
    usort($backups, function($a, $b) {
        return $b['created'] - $a['created'];
    });
}

include 'includes/header.php';
?>

<div class="space-y-8">
    <div>
        <h1 class="text-3xl font-bold text-gray-900">Backup & Restore</h1>
        <p class="mt-2 text-gray-600">Manage your database backups</p>
    </div>

    <?php if (isset($error)): ?>
    <div class="bg-red-50 border border-red-200 text-red-700 px-4 py-3 rounded-lg">
        <i class="fas fa-exclamation-circle mr-2"></i>
        <?php echo htmlspecialchars($error); ?>
    </div>
    <?php endif; ?>

    <?php if (isset($success)): ?>
    <div class="bg-green-50 border border-green-200 text-green-700 px-4 py-3 rounded-lg">
        <i class="fas fa-check-circle mr-2"></i>
        <?php echo htmlspecialchars($success); ?>
    </div>
    <?php endif; ?>

    <!-- Create Backup -->
    <div class="bg-white rounded-xl shadow-sm border border-gray-100 p-6">
        <div class="flex items-center justify-between">
            <div>
                <h2 class="text-lg font-semibold text-gray-900">Create New Backup</h2>
                <p class="text-sm text-gray-600 mt-1">Create a complete backup of your database</p>
            </div>
            <form method="POST" class="inline">
                <input type="hidden" name="csrf_token" value="<?php echo $auth->generateCSRFToken(); ?>">
                <button
                    type="submit"
                    name="create_backup"
                    class="bg-blue-600 text-white px-4 py-2 rounded-lg hover:bg-blue-700 flex items-center space-x-2 transition-colors"
                >
                    <i class="fas fa-download"></i>
                    <span>Create Backup</span>
                </button>
            </form>
        </div>
    </div>

    <!-- Existing Backups -->
    <div class="bg-white rounded-xl shadow-sm border border-gray-100">
        <div class="p-6 border-b border-gray-100">
            <h2 class="text-lg font-semibold text-gray-900">Existing Backups</h2>
        </div>
        
        <?php if (!empty($backups)): ?>
        <div class="overflow-x-auto">
            <table class="min-w-full divide-y divide-gray-200">
                <thead class="bg-gray-50">
                    <tr>
                        <th class="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider">
                            Filename
                        </th>
                        <th class="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider">
                            Size
                        </th>
                        <th class="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider">
                            Created
                        </th>
                        <th class="px-6 py-3 text-right text-xs font-medium text-gray-500 uppercase tracking-wider">
                            Actions
                        </th>
                    </tr>
                </thead>
                <tbody class="bg-white divide-y divide-gray-200">
                    <?php foreach ($backups as $backup): ?>
                    <tr class="hover:bg-gray-50">
                        <td class="px-6 py-4 whitespace-nowrap">
                            <div class="flex items-center">
                                <i class="fas fa-file-archive text-blue-600 mr-3"></i>
                                <span class="text-sm font-medium text-gray-900"><?php echo htmlspecialchars($backup['filename']); ?></span>
                            </div>
                        </td>
                        <td class="px-6 py-4 whitespace-nowrap text-sm text-gray-500">
                            <?php echo formatFileSize($backup['size']); ?>
                        </td>
                        <td class="px-6 py-4 whitespace-nowrap text-sm text-gray-500">
                            <?php echo formatDate(date('Y-m-d H:i:s', $backup['created']), 'M d, Y H:i'); ?>
                        </td>
                        <td class="px-6 py-4 whitespace-nowrap text-right text-sm font-medium">
                            <div class="flex items-center justify-end space-x-2">
                                <a
                                    href="<?php echo $backupDir . $backup['filename']; ?>"
                                    download
                                    class="text-blue-600 hover:text-blue-900 p-1 rounded"
                                    title="Download"
                                >
                                    <i class="fas fa-download"></i>
                                </a>
                                <button
                                    onclick="deleteBackup('<?php echo htmlspecialchars($backup['filename']); ?>')"
                                    class="text-red-600 hover:text-red-900 p-1 rounded"
                                    title="Delete"
                                >
                                    <i class="fas fa-trash"></i>
                                </button>
                            </div>
                        </td>
                    </tr>
                    <?php endforeach; ?>
                </tbody>
            </table>
        </div>
        <?php else: ?>
        <div class="text-center py-12">
            <i class="fas fa-archive text-4xl text-gray-400 mb-4"></i>
            <p class="text-gray-500">No backups found</p>
            <p class="text-sm text-gray-400">Create your first backup to secure your data</p>
        </div>
        <?php endif; ?>
    </div>

    <!-- Backup Instructions -->
    <div class="bg-blue-50 border border-blue-200 rounded-xl p-6">
        <div class="flex items-start">
            <i class="fas fa-info-circle text-blue-600 mt-1 mr-3"></i>
            <div>
                <h3 class="text-sm font-medium text-blue-800 mb-2">Backup Instructions</h3>
                <ul class="text-sm text-blue-700 space-y-1">
                    <li>• Create regular backups to protect your data</li>
                    <li>• Store backups in a secure location outside your web directory</li>
                    <li>• Test restore procedures periodically</li>
                    <li>• Keep multiple backup copies for different time periods</li>
                </ul>
            </div>
        </div>
    </div>
</div>

<script>
function deleteBackup(filename) {
    if (confirm(`Are you sure you want to delete backup "${filename}"?`)) {
        const form = document.createElement('form');
        form.method = 'POST';
        form.innerHTML = `
            <input type="hidden" name="csrf_token" value="<?php echo $auth->generateCSRFToken(); ?>">
            <input type="hidden" name="delete_backup" value="1">
            <input type="hidden" name="filename" value="${filename}">
        `;
        document.body.appendChild(form);
        form.submit();
    }
}
</script>

<?php
function formatFileSize($bytes) {
    if ($bytes >= 1073741824) {
        return number_format($bytes / 1073741824, 2) . ' GB';
    } elseif ($bytes >= 1048576) {
        return number_format($bytes / 1048576, 2) . ' MB';
    } elseif ($bytes >= 1024) {
        return number_format($bytes / 1024, 2) . ' KB';
    } else {
        return $bytes . ' bytes';
    }
}
?>

</main>
</body>
</html>