<?php
require_once '../config/config.php';
require_once '../config/database.php';
require_once '../includes/auth.php';
require_once '../includes/functions.php';

header('Content-Type: application/json');

$auth->requireAdmin();

$method = $_SERVER['REQUEST_METHOD'];
$action = $_GET['action'] ?? $_POST['action'] ?? '';

try {
    switch ($method) {
        case 'GET':
            if ($action === 'list') {
                $stmt = $db->prepare("SELECT id, email, full_name, role, created_at FROM users ORDER BY created_at DESC");
                $stmt->execute();
                $users = $stmt->fetchAll();
                sendJsonResponse(['success' => true, 'data' => $users]);
            } else {
                sendJsonResponse(['success' => false, 'message' => 'Invalid action'], 400);
            }
            break;

        case 'POST':
            if (!$auth->validateCSRFToken($_POST['csrf_token'] ?? '')) {
                sendJsonResponse(['success' => false, 'message' => 'CSRF token validation failed'], 403);
            }

            if ($action === 'create') {
                $fullName = sanitizeInput($_POST['full_name'] ?? '');
                $email = sanitizeInput($_POST['email'] ?? '');
                $password = $_POST['password'] ?? '';
                $role = sanitizeInput($_POST['role'] ?? 'cashier');

                if (empty($fullName) || empty($email) || empty($password)) {
                    sendJsonResponse(['success' => false, 'message' => 'All fields required'], 400);
                }

                if (!validateEmail($email)) {
                    sendJsonResponse(['success' => false, 'message' => 'Invalid email format'], 400);
                }

                if (!in_array($role, ['admin', 'cashier'])) {
                    sendJsonResponse(['success' => false, 'message' => 'Invalid role'], 400);
                }

                // Check if email exists
                $stmt = $db->prepare("SELECT id FROM users WHERE email = ?");
                $stmt->execute([$email]);
                if ($stmt->fetch()) {
                    sendJsonResponse(['success' => false, 'message' => 'Email already exists'], 400);
                }

                $hashedPassword = password_hash($password, PASSWORD_DEFAULT);
                $stmt = $db->prepare("
                    INSERT INTO users (email, password, full_name, role)
                    VALUES (?, ?, ?, ?)
                ");
                $stmt->execute([$email, $hashedPassword, $fullName, $role]);
                
                sendJsonResponse(['success' => true, 'message' => 'User created successfully']);

            } elseif ($action === 'update') {
                $userId = intval($_POST['user_id'] ?? 0);
                $fullName = sanitizeInput($_POST['full_name'] ?? '');
                $role = sanitizeInput($_POST['role'] ?? 'cashier');
                $password = $_POST['password'] ?? '';

                if (!$userId || empty($fullName)) {
                    sendJsonResponse(['success' => false, 'message' => 'User ID and name required'], 400);
                }

                if (!in_array($role, ['admin', 'cashier'])) {
                    sendJsonResponse(['success' => false, 'message' => 'Invalid role'], 400);
                }

                if (!empty($password)) {
                    $hashedPassword = password_hash($password, PASSWORD_DEFAULT);
                    $stmt = $db->prepare("
                        UPDATE users 
                        SET full_name = ?, role = ?, password = ?
                        WHERE id = ?
                    ");
                    $stmt->execute([$fullName, $role, $hashedPassword, $userId]);
                } else {
                    $stmt = $db->prepare("
                        UPDATE users 
                        SET full_name = ?, role = ?
                        WHERE id = ?
                    ");
                    $stmt->execute([$fullName, $role, $userId]);
                }
                
                sendJsonResponse(['success' => true, 'message' => 'User updated successfully']);

            } elseif ($action === 'delete') {
                $userId = intval($_POST['user_id'] ?? 0);
                
                if (!$userId) {
                    sendJsonResponse(['success' => false, 'message' => 'User ID required'], 400);
                }

                if ($userId == $_SESSION['user_id']) {
                    sendJsonResponse(['success' => false, 'message' => 'Cannot delete your own account'], 400);
                }

                $stmt = $db->prepare("DELETE FROM users WHERE id = ?");
                $stmt->execute([$userId]);
                
                sendJsonResponse(['success' => true, 'message' => 'User deleted successfully']);
            }
            break;

        default:
            sendJsonResponse(['success' => false, 'message' => 'Method not allowed'], 405);
    }
} catch (Exception $e) {
    logError("Users API error: " . $e->getMessage());
    sendJsonResponse(['success' => false, 'message' => 'Internal server error'], 500);
}
?>