<?php
require_once '../config/config.php';
require_once '../config/database.php';
require_once '../includes/auth.php';
require_once '../includes/functions.php';

header('Content-Type: application/json');

$auth->requireAdmin();

$method = $_SERVER['REQUEST_METHOD'];

try {
    if ($method === 'GET') {
        // Get current settings
        $settings = getShopSettings($db);
        sendJsonResponse(['success' => true, 'data' => $settings]);

    } elseif ($method === 'POST') {
        if (!$auth->validateCSRFToken($_POST['csrf_token'] ?? '')) {
            sendJsonResponse(['success' => false, 'message' => 'CSRF token validation failed'], 403);
        }

        $shopName = sanitizeInput($_POST['shop_name'] ?? '');
        $address = sanitizeInput($_POST['address'] ?? '');
        $phoneNumber = sanitizeInput($_POST['phone_number'] ?? '');
        $email = sanitizeInput($_POST['email'] ?? '');
        $mpesaTillNumber = sanitizeInput($_POST['mpesa_till_number'] ?? '');
        $taxRate = floatval($_POST['tax_rate'] ?? 16);
        $receiptHeader = sanitizeInput($_POST['receipt_header'] ?? '');
        $receiptFooter = sanitizeInput($_POST['receipt_footer'] ?? '');
        $logoUrl = sanitizeInput($_POST['logo_url'] ?? '');

        if (empty($shopName) || empty($address) || empty($phoneNumber)) {
            sendJsonResponse(['success' => false, 'message' => 'Required fields missing'], 400);
        }

        if (!empty($email) && !validateEmail($email)) {
            sendJsonResponse(['success' => false, 'message' => 'Invalid email format'], 400);
        }

        if ($taxRate < 0 || $taxRate > 100) {
            sendJsonResponse(['success' => false, 'message' => 'Tax rate must be between 0 and 100'], 400);
        }

        // Check if settings exist
        $stmt = $db->prepare("SELECT id FROM shop_settings LIMIT 1");
        $stmt->execute();
        $existingSettings = $stmt->fetch();

        $settingsData = [
            $shopName,
            $address,
            $phoneNumber,
            $email ?: null,
            $mpesaTillNumber ?: null,
            $taxRate,
            $receiptHeader ?: null,
            $receiptFooter ?: null,
            $logoUrl ?: null
        ];

        if ($existingSettings) {
            // Update existing settings
            $stmt = $db->prepare("
                UPDATE shop_settings 
                SET shop_name = ?, address = ?, phone_number = ?, email = ?, 
                    mpesa_till_number = ?, tax_rate = ?, receipt_header = ?, 
                    receipt_footer = ?, logo_url = ?
                WHERE id = ?
            ");
            $settingsData[] = $existingSettings['id'];
            $stmt->execute($settingsData);
        } else {
            // Insert new settings
            $stmt = $db->prepare("
                INSERT INTO shop_settings (
                    shop_name, address, phone_number, email, mpesa_till_number,
                    tax_rate, receipt_header, receipt_footer, logo_url
                ) VALUES (?, ?, ?, ?, ?, ?, ?, ?, ?)
            ");
            $stmt->execute($settingsData);
        }

        sendJsonResponse(['success' => true, 'message' => 'Settings saved successfully']);

    } else {
        sendJsonResponse(['success' => false, 'message' => 'Method not allowed'], 405);
    }
} catch (Exception $e) {
    logError("Settings API error: " . $e->getMessage());
    sendJsonResponse(['success' => false, 'message' => 'Internal server error'], 500);
}
?>