<?php
require_once '../config/config.php';
require_once '../config/database.php';
require_once '../includes/auth.php';
require_once '../includes/functions.php';

header('Content-Type: application/json');

$auth->requireLogin();

if ($_SERVER['REQUEST_METHOD'] !== 'POST') {
    sendJsonResponse(['success' => false, 'message' => 'Method not allowed'], 405);
}

try {
    $input = json_decode(file_get_contents('php://input'), true);
    
    if (!$input) {
        sendJsonResponse(['success' => false, 'message' => 'Invalid JSON data'], 400);
    }

    $customerName = sanitizeInput($input['customer_name'] ?? '');
    $paymentMethod = sanitizeInput($input['payment_method'] ?? 'cash');
    $paymentReference = sanitizeInput($input['payment_reference'] ?? '');
    $discountAmount = floatval($input['discount_amount'] ?? 0);
    $cartItems = $input['cart_items'] ?? [];

    if (empty($cartItems)) {
        sendJsonResponse(['success' => false, 'message' => 'Cart is empty'], 400);
    }

    // Validate payment method
    if (!in_array($paymentMethod, ['cash', 'mpesa', 'card'])) {
        sendJsonResponse(['success' => false, 'message' => 'Invalid payment method'], 400);
    }

    // Calculate totals
    $subtotal = 0;
    foreach ($cartItems as $item) {
        $subtotal += floatval($item['unit_price']) * intval($item['quantity']);
    }

    $shopSettings = getShopSettings($db);
    $taxRate = $shopSettings ? floatval($shopSettings['tax_rate']) : 0;
    $taxAmount = $subtotal * ($taxRate / 100);
    $totalAmount = $subtotal + $taxAmount - $discountAmount;

    if ($totalAmount < 0) {
        sendJsonResponse(['success' => false, 'message' => 'Total amount cannot be negative'], 400);
    }

    // Start transaction
    $db->beginTransaction();

    try {
        // Generate receipt number
        $receiptNumber = generateReceiptNumber();

        // Insert sale record
        $stmt = $db->prepare("
            INSERT INTO sales (
                receipt_number, customer_name, subtotal, tax_amount, 
                discount_amount, total_amount, payment_method, 
                payment_reference, cashier_id
            ) VALUES (?, ?, ?, ?, ?, ?, ?, ?, ?)
        ");
        
        $stmt->execute([
            $receiptNumber,
            $customerName ?: null,
            $subtotal,
            $taxAmount,
            $discountAmount,
            $totalAmount,
            $paymentMethod,
            $paymentReference ?: null,
            $_SESSION['user_id']
        ]);

        $saleId = $db->lastInsertId();

        // Insert sale items
        $stmt = $db->prepare("
            INSERT INTO sale_items (sale_id, service_id, quantity, unit_price, total_price)
            VALUES (?, ?, ?, ?, ?)
        ");

        foreach ($cartItems as $item) {
            $quantity = intval($item['quantity']);
            $unitPrice = floatval($item['unit_price']);
            $totalPrice = $unitPrice * $quantity;

            $stmt->execute([
                $saleId,
                $item['id'],
                $quantity,
                $unitPrice,
                $totalPrice
            ]);
        }

        // Commit transaction
        $db->commit();

        sendJsonResponse([
            'success' => true,
            'message' => 'Sale completed successfully',
            'sale_id' => $saleId,
            'receipt_number' => $receiptNumber
        ]);

    } catch (Exception $e) {
        $db->rollBack();
        throw $e;
    }

} catch (Exception $e) {
    logError("Error processing sale: " . $e->getMessage());
    sendJsonResponse(['success' => false, 'message' => 'Failed to process sale'], 500);
}
?>