<?php
require_once '../config/config.php';
require_once '../config/database.php';
require_once '../includes/auth.php';
require_once '../includes/functions.php';

header('Content-Type: application/json');

$auth->requireAdmin();

$method = $_SERVER['REQUEST_METHOD'];
$action = $_GET['action'] ?? '';

try {
    if ($method === 'GET' && $action === 'dashboard') {
        // Get dashboard statistics
        $today = date('Y-m-d');
        
        // Today's sales
        $stmt = $db->prepare("
            SELECT 
                COUNT(*) as transaction_count,
                COALESCE(SUM(total_amount), 0) as total_revenue
            FROM sales 
            WHERE DATE(created_at) = ?
        ");
        $stmt->execute([$today]);
        $todayStats = $stmt->fetch();

        // Top services today
        $stmt = $db->prepare("
            SELECT 
                s.name as service_name,
                SUM(si.quantity) as total_quantity,
                SUM(si.total_price) as total_revenue
            FROM sale_items si
            JOIN services s ON si.service_id = s.id
            JOIN sales sa ON si.sale_id = sa.id
            WHERE DATE(sa.created_at) = ?
            GROUP BY s.id, s.name
            ORDER BY total_revenue DESC
            LIMIT 5
        ");
        $stmt->execute([$today]);
        $topServices = $stmt->fetchAll();

        // Cashier performance today
        $stmt = $db->prepare("
            SELECT 
                u.full_name as cashier_name,
                COUNT(s.id) as transaction_count,
                COALESCE(SUM(s.total_amount), 0) as total_sales
            FROM sales s
            JOIN users u ON s.cashier_id = u.id
            WHERE DATE(s.created_at) = ?
            GROUP BY u.id, u.full_name
            ORDER BY total_sales DESC
        ");
        $stmt->execute([$today]);
        $cashierPerformance = $stmt->fetchAll();

        sendJsonResponse([
            'success' => true,
            'data' => [
                'today_stats' => $todayStats,
                'top_services' => $topServices,
                'cashier_performance' => $cashierPerformance
            ]
        ]);

    } elseif ($method === 'GET' && $action === 'sales') {
        $dateRange = $_GET['range'] ?? 'today';
        $startDate = $_GET['start_date'] ?? date('Y-m-d');
        $endDate = $_GET['end_date'] ?? date('Y-m-d');

        // Calculate date range
        switch ($dateRange) {
            case 'today':
                $start = date('Y-m-d 00:00:00');
                $end = date('Y-m-d 23:59:59');
                break;
            case 'yesterday':
                $start = date('Y-m-d 00:00:00', strtotime('-1 day'));
                $end = date('Y-m-d 23:59:59', strtotime('-1 day'));
                break;
            case 'week':
                $start = date('Y-m-d 00:00:00', strtotime('monday this week'));
                $end = date('Y-m-d 23:59:59', strtotime('sunday this week'));
                break;
            case 'month':
                $start = date('Y-m-01 00:00:00');
                $end = date('Y-m-t 23:59:59');
                break;
            case 'custom':
                $start = $startDate . ' 00:00:00';
                $end = $endDate . ' 23:59:59';
                break;
            default:
                $start = date('Y-m-d 00:00:00');
                $end = date('Y-m-d 23:59:59');
        }

        // Get sales summary
        $stmt = $db->prepare("
            SELECT 
                COUNT(*) as total_transactions,
                COALESCE(SUM(total_amount), 0) as total_revenue,
                COALESCE(AVG(total_amount), 0) as avg_transaction
            FROM sales 
            WHERE created_at BETWEEN ? AND ?
        ");
        $stmt->execute([$start, $end]);
        $summary = $stmt->fetch();

        // Get service breakdown
        $stmt = $db->prepare("
            SELECT 
                srv.name as service_name,
                SUM(si.quantity) as total_quantity,
                SUM(si.total_price) as total_revenue
            FROM sale_items si
            JOIN services srv ON si.service_id = srv.id
            JOIN sales s ON si.sale_id = s.id
            WHERE s.created_at BETWEEN ? AND ?
            GROUP BY srv.id, srv.name
            ORDER BY total_revenue DESC
        ");
        $stmt->execute([$start, $end]);
        $serviceBreakdown = $stmt->fetchAll();

        // Get recent transactions
        $stmt = $db->prepare("
            SELECT s.*, u.full_name as cashier_name
            FROM sales s
            JOIN users u ON s.cashier_id = u.id
            WHERE s.created_at BETWEEN ? AND ?
            ORDER BY s.created_at DESC
            LIMIT 20
        ");
        $stmt->execute([$start, $end]);
        $recentTransactions = $stmt->fetchAll();

        sendJsonResponse([
            'success' => true,
            'data' => [
                'summary' => $summary,
                'service_breakdown' => $serviceBreakdown,
                'recent_transactions' => $recentTransactions
            ]
        ]);

    } else {
        sendJsonResponse(['success' => false, 'message' => 'Invalid request'], 400);
    }
} catch (Exception $e) {
    logError("Reports API error: " . $e->getMessage());
    sendJsonResponse(['success' => false, 'message' => 'Internal server error'], 500);
}
?>