<?php
require_once '../config/config.php';
require_once '../config/database.php';
require_once '../includes/auth.php';
require_once '../includes/functions.php';

header('Content-Type: application/json');

$auth->requireAdmin();

$method = $_SERVER['REQUEST_METHOD'];
$action = $_GET['action'] ?? $_POST['action'] ?? '';

try {
    switch ($method) {
        case 'GET':
            if ($action === 'list') {
                $stmt = $db->prepare("
                    SELECT c.*, COUNT(s.id) as service_count 
                    FROM categories c 
                    LEFT JOIN services s ON c.id = s.category_id 
                    GROUP BY c.id 
                    ORDER BY c.name
                ");
                $stmt->execute();
                $categories = $stmt->fetchAll();
                sendJsonResponse(['success' => true, 'data' => $categories]);
            } else {
                sendJsonResponse(['success' => false, 'message' => 'Invalid action'], 400);
            }
            break;

        case 'POST':
            if (!$auth->validateCSRFToken($_POST['csrf_token'] ?? '')) {
                sendJsonResponse(['success' => false, 'message' => 'CSRF token validation failed'], 403);
            }

            if ($action === 'create') {
                $name = sanitizeInput($_POST['name'] ?? '');
                $description = sanitizeInput($_POST['description'] ?? '');

                if (empty($name)) {
                    sendJsonResponse(['success' => false, 'message' => 'Category name required'], 400);
                }

                $stmt = $db->prepare("INSERT INTO categories (name, description) VALUES (?, ?)");
                $stmt->execute([$name, $description]);
                
                sendJsonResponse(['success' => true, 'message' => 'Category created successfully']);

            } elseif ($action === 'update') {
                $categoryId = intval($_POST['category_id'] ?? 0);
                $name = sanitizeInput($_POST['name'] ?? '');
                $description = sanitizeInput($_POST['description'] ?? '');

                if (!$categoryId || empty($name)) {
                    sendJsonResponse(['success' => false, 'message' => 'Invalid data provided'], 400);
                }

                $stmt = $db->prepare("UPDATE categories SET name = ?, description = ? WHERE id = ?");
                $stmt->execute([$name, $description, $categoryId]);
                
                sendJsonResponse(['success' => true, 'message' => 'Category updated successfully']);

            } elseif ($action === 'delete') {
                $categoryId = intval($_POST['category_id'] ?? 0);
                
                if (!$categoryId) {
                    sendJsonResponse(['success' => false, 'message' => 'Category ID required'], 400);
                }

                // Check if category has services
                $stmt = $db->prepare("SELECT COUNT(*) as count FROM services WHERE category_id = ?");
                $stmt->execute([$categoryId]);
                $result = $stmt->fetch();
                
                if ($result['count'] > 0) {
                    sendJsonResponse(['success' => false, 'message' => 'Cannot delete category with existing services'], 400);
                }

                $stmt = $db->prepare("DELETE FROM categories WHERE id = ?");
                $stmt->execute([$categoryId]);
                
                sendJsonResponse(['success' => true, 'message' => 'Category deleted successfully']);
            }
            break;

        default:
            sendJsonResponse(['success' => false, 'message' => 'Method not allowed'], 405);
    }
} catch (Exception $e) {
    logError("Categories API error: " . $e->getMessage());
    sendJsonResponse(['success' => false, 'message' => 'Internal server error'], 500);
}
?>