# Cyber Shop POS System - Deployment Guide

## Production Deployment Checklist

### 1. Server Requirements
- **PHP**: 7.4 or higher (8.0+ recommended)
- **MySQL**: 5.7 or higher (8.0+ recommended)
- **Web Server**: Apache 2.4+ or Nginx 1.18+
- **SSL Certificate**: Required for production
- **Memory**: Minimum 512MB RAM
- **Storage**: Minimum 1GB free space

### 2. Pre-Deployment Setup

#### A. Domain and Hosting
```bash
# Recommended hosting providers for Kenya:
# - Safaricom Cloud
# - Kenya Web Experts
# - Truehost Kenya
# - Hostpinnacle
```

#### B. SSL Certificate
```bash
# Install Let's Encrypt SSL (free)
sudo apt install certbot python3-certbot-apache
sudo certbot --apache -d yourdomain.com
```

#### C. Database Preparation
```sql
-- Create production database
CREATE DATABASE cyber_pos_prod CHARACTER SET utf8mb4 COLLATE utf8mb4_unicode_ci;

-- Create dedicated user
CREATE USER 'cyber_pos_prod'@'localhost' IDENTIFIED BY 'STRONG_RANDOM_PASSWORD';
GRANT ALL PRIVILEGES ON cyber_pos_prod.* TO 'cyber_pos_prod'@'localhost';
FLUSH PRIVILEGES;
```

### 3. File Upload and Configuration

#### A. Upload Files
```bash
# Upload via FTP/SFTP to web root
# Recommended structure:
/var/www/html/
├── cyber-pos/
│   ├── config/
│   ├── includes/
│   ├── api/
│   ├── assets/
│   └── ...
```

#### B. Set File Permissions
```bash
# Set proper permissions
find /var/www/html/cyber-pos -type d -exec chmod 755 {} \;
find /var/www/html/cyber-pos -type f -exec chmod 644 {} \;

# Secure sensitive files
chmod 600 /var/www/html/cyber-pos/config/config.php
chmod 755 /var/www/html/cyber-pos/receipts/
chmod 755 /var/www/html/cyber-pos/backups/
chmod 755 /var/www/html/cyber-pos/logs/
```

#### C. Configuration
```php
// config/config.php - Production settings
define('DB_HOST', 'localhost');
define('DB_NAME', 'cyber_pos_prod');
define('DB_USER', 'cyber_pos_prod');
define('DB_PASS', 'YOUR_SECURE_PASSWORD');

define('APP_URL', 'https://yourdomain.com/cyber-pos');

// Disable error display in production
error_reporting(0);
ini_set('display_errors', 0);
```

### 4. Database Import
```bash
# Import schema
mysql -u cyber_pos_prod -p cyber_pos_prod < database/schema.sql

# Import sample data (optional)
mysql -u cyber_pos_prod -p cyber_pos_prod < database/sample_data.sql

# Import additional tables
mysql -u cyber_pos_prod -p cyber_pos_prod < database/inventory_schema.sql
mysql -u cyber_pos_prod -p cyber_pos_prod < database/mpesa_schema.sql
```

### 5. Web Server Configuration

#### Apache Virtual Host
```apache
<VirtualHost *:443>
    ServerName yourdomain.com
    DocumentRoot /var/www/html/cyber-pos
    
    SSLEngine on
    SSLCertificateFile /etc/letsencrypt/live/yourdomain.com/fullchain.pem
    SSLCertificateKeyFile /etc/letsencrypt/live/yourdomain.com/privkey.pem
    
    <Directory /var/www/html/cyber-pos>
        AllowOverride All
        Require all granted
    </Directory>
    
    # Security headers
    Header always set X-Content-Type-Options nosniff
    Header always set X-Frame-Options DENY
    Header always set X-XSS-Protection "1; mode=block"
    
    # Error and access logs
    ErrorLog ${APACHE_LOG_DIR}/cyber-pos-error.log
    CustomLog ${APACHE_LOG_DIR}/cyber-pos-access.log combined
</VirtualHost>

# Redirect HTTP to HTTPS
<VirtualHost *:80>
    ServerName yourdomain.com
    Redirect permanent / https://yourdomain.com/
</VirtualHost>
```

#### Nginx Configuration
```nginx
server {
    listen 443 ssl http2;
    server_name yourdomain.com;
    root /var/www/html/cyber-pos;
    index index.php;

    ssl_certificate /etc/letsencrypt/live/yourdomain.com/fullchain.pem;
    ssl_certificate_key /etc/letsencrypt/live/yourdomain.com/privkey.pem;

    # Security headers
    add_header X-Content-Type-Options nosniff;
    add_header X-Frame-Options DENY;
    add_header X-XSS-Protection "1; mode=block";

    location / {
        try_files $uri $uri/ /index.php?$query_string;
    }

    location ~ \.php$ {
        fastcgi_pass unix:/var/run/php/php8.0-fpm.sock;
        fastcgi_index index.php;
        fastcgi_param SCRIPT_FILENAME $document_root$fastcgi_script_name;
        include fastcgi_params;
    }

    # Deny access to sensitive files
    location ~ /\. {
        deny all;
    }

    location ~ /(config|includes|logs|backups)/ {
        deny all;
    }
}

# Redirect HTTP to HTTPS
server {
    listen 80;
    server_name yourdomain.com;
    return 301 https://$server_name$request_uri;
}
```

### 6. Security Hardening

#### A. File Security
```bash
# Create secure directories
mkdir -p /var/www/html/cyber-pos/logs
mkdir -p /var/www/html/cyber-pos/backups
mkdir -p /var/www/html/cyber-pos/uploads

# Set ownership
chown -R www-data:www-data /var/www/html/cyber-pos
```

#### B. Database Security
```sql
-- Remove test accounts in production
DELETE FROM users WHERE email IN ('admin@cybershop.ke', 'cashier@cybershop.ke');

-- Create your actual admin account
INSERT INTO users (email, password, full_name, role) VALUES
('your-email@domain.com', '$2y$10$...', 'Your Name', 'admin');
```

#### C. PHP Security
```ini
; php.ini security settings
expose_php = Off
display_errors = Off
log_errors = On
error_log = /var/log/php_errors.log
session.cookie_secure = 1
session.cookie_httponly = 1
session.use_strict_mode = 1
```

### 7. Backup Strategy

#### A. Automated Daily Backups
```bash
#!/bin/bash
# /etc/cron.daily/cyber-pos-backup

DATE=$(date +%Y%m%d_%H%M%S)
BACKUP_DIR="/var/backups/cyber-pos"
DB_NAME="cyber_pos_prod"
DB_USER="cyber_pos_prod"
DB_PASS="YOUR_PASSWORD"

# Create backup directory
mkdir -p $BACKUP_DIR

# Database backup
mysqldump -u $DB_USER -p$DB_PASS $DB_NAME > $BACKUP_DIR/db_backup_$DATE.sql

# Files backup
tar -czf $BACKUP_DIR/files_backup_$DATE.tar.gz /var/www/html/cyber-pos

# Keep only last 30 days
find $BACKUP_DIR -name "*.sql" -mtime +30 -delete
find $BACKUP_DIR -name "*.tar.gz" -mtime +30 -delete

# Log backup completion
echo "$(date): Backup completed successfully" >> /var/log/cyber-pos-backup.log
```

#### B. Make Backup Script Executable
```bash
chmod +x /etc/cron.daily/cyber-pos-backup
```

### 8. Monitoring and Maintenance

#### A. Log Rotation
```bash
# /etc/logrotate.d/cyber-pos
/var/www/html/cyber-pos/logs/*.log {
    daily
    rotate 30
    compress
    missingok
    notifempty
    create 644 www-data www-data
}
```

#### B. Health Check Script
```bash
#!/bin/bash
# /usr/local/bin/cyber-pos-health-check

# Check if web server is responding
if ! curl -f -s https://yourdomain.com/cyber-pos/ > /dev/null; then
    echo "$(date): Website not responding" >> /var/log/cyber-pos-health.log
    # Send alert email or SMS
fi

# Check database connection
if ! mysql -u cyber_pos_prod -p$DB_PASS -e "SELECT 1" cyber_pos_prod > /dev/null 2>&1; then
    echo "$(date): Database connection failed" >> /var/log/cyber-pos-health.log
    # Send alert
fi

# Check disk space
DISK_USAGE=$(df /var/www/html | tail -1 | awk '{print $5}' | sed 's/%//')
if [ $DISK_USAGE -gt 90 ]; then
    echo "$(date): Disk usage high: ${DISK_USAGE}%" >> /var/log/cyber-pos-health.log
fi
```

### 9. Performance Optimization

#### A. PHP Optimization
```ini
; php.ini performance settings
opcache.enable=1
opcache.memory_consumption=128
opcache.max_accelerated_files=4000
opcache.revalidate_freq=60
```

#### B. MySQL Optimization
```sql
-- Add indexes for better performance
ALTER TABLE sales ADD INDEX idx_created_at (created_at);
ALTER TABLE sales ADD INDEX idx_cashier_date (cashier_id, created_at);
ALTER TABLE sale_items ADD INDEX idx_sale_service (sale_id, service_id);
```

### 10. M-Pesa Integration (Production)

#### A. Safaricom API Setup
1. Register at [developer.safaricom.co.ke](https://developer.safaricom.co.ke)
2. Create production app
3. Get production credentials
4. Configure in M-Pesa Integration page

#### B. Callback URL Setup
```php
// api/mpesa_callback.php
<?php
// Handle M-Pesa payment callbacks
$input = file_get_contents('php://input');
$data = json_decode($input, true);

// Process callback and update transaction status
// Log all callbacks for debugging
file_put_contents('logs/mpesa_callbacks.log', date('Y-m-d H:i:s') . ": " . $input . "\n", FILE_APPEND);
?>
```

### 11. Go-Live Checklist

#### Before Launch:
- [ ] Change all default passwords
- [ ] Configure shop settings with real information
- [ ] Add actual services and pricing
- [ ] Create real user accounts
- [ ] Test all functionality thoroughly
- [ ] Configure SSL certificate
- [ ] Set up automated backups
- [ ] Configure monitoring

#### After Launch:
- [ ] Monitor error logs daily
- [ ] Check backup completion
- [ ] Review sales reports
- [ ] Train staff on system usage
- [ ] Document any customizations
- [ ] Plan regular maintenance

### 12. Maintenance Schedule

#### Daily:
- Check error logs
- Verify backup completion
- Monitor system performance

#### Weekly:
- Review sales reports
- Check user accounts
- Update service pricing if needed

#### Monthly:
- Full system backup verification
- Security updates check
- Performance optimization review

### 13. Support and Updates

#### Getting Help:
- Check built-in help documentation
- Review error logs for issues
- Test in staging environment first
- Document any problems clearly

#### Future Updates:
- Always backup before updates
- Test updates in staging environment
- Plan updates during low-traffic periods
- Keep documentation updated

This deployment guide ensures your cyber shop POS system runs securely and efficiently in production with proper monitoring, backups, and maintenance procedures.