import React, { useState, useEffect } from 'react';
import { TrendingUp, Users, ShoppingCart, DollarSign } from 'lucide-react';
import { supabase } from '../lib/supabase';
import { DashboardStats } from '../types';
import { format } from 'date-fns';

export const Dashboard: React.FC = () => {
  const [stats, setStats] = useState<DashboardStats | null>(null);
  const [loading, setLoading] = useState(true);

  useEffect(() => {
    fetchDashboardStats();
  }, []);

  const fetchDashboardStats = async () => {
    try {
      const today = format(new Date(), 'yyyy-MM-dd');
      
      // Today's sales
      const { data: todaySales } = await supabase
        .from('sales')
        .select('total_amount')
        .gte('created_at', `${today}T00:00:00`)
        .lt('created_at', `${today}T23:59:59`);

      // Today's transaction count
      const { data: todayTransactions } = await supabase
        .from('sales')
        .select('id')
        .gte('created_at', `${today}T00:00:00`)
        .lt('created_at', `${today}T23:59:59`);

      // Top services
      const { data: topServices } = await supabase
        .from('sale_items')
        .select(`
          service:services(name),
          total_price,
          quantity
        `)
        .gte('created_at', `${today}T00:00:00`)
        .lt('created_at', `${today}T23:59:59`);

      // Cashier performance
      const { data: cashierPerformance } = await supabase
        .from('sales')
        .select(`
          total_amount,
          cashier:users(full_name)
        `)
        .gte('created_at', `${today}T00:00:00`)
        .lt('created_at', `${today}T23:59:59`);

      const todaySalesTotal = todaySales?.reduce((sum, sale) => sum + sale.total_amount, 0) || 0;
      const todayTransactionCount = todayTransactions?.length || 0;

      // Process top services
      const serviceStats = topServices?.reduce((acc: any, item: any) => {
        const serviceName = item.service?.name || 'Unknown';
        if (!acc[serviceName]) {
          acc[serviceName] = { total_sales: 0, transaction_count: 0 };
        }
        acc[serviceName].total_sales += item.total_price;
        acc[serviceName].transaction_count += item.quantity;
        return acc;
      }, {});

      const topServicesArray = Object.entries(serviceStats || {})
        .map(([service_name, stats]: [string, any]) => ({
          service_name,
          total_sales: stats.total_sales,
          transaction_count: stats.transaction_count,
        }))
        .sort((a, b) => b.total_sales - a.total_sales)
        .slice(0, 5);

      // Process cashier performance
      const cashierStats = cashierPerformance?.reduce((acc: any, sale: any) => {
        const cashierName = sale.cashier?.full_name || 'Unknown';
        if (!acc[cashierName]) {
          acc[cashierName] = { total_sales: 0, transaction_count: 0 };
        }
        acc[cashierName].total_sales += sale.total_amount;
        acc[cashierName].transaction_count += 1;
        return acc;
      }, {});

      const cashierPerformanceArray = Object.entries(cashierStats || {})
        .map(([cashier_name, stats]: [string, any]) => ({
          cashier_name,
          total_sales: stats.total_sales,
          transaction_count: stats.transaction_count,
        }))
        .sort((a, b) => b.total_sales - a.total_sales);

      setStats({
        todaySales: todaySalesTotal,
        todayTransactions: todayTransactionCount,
        topServices: topServicesArray,
        cashierPerformance: cashierPerformanceArray,
      });
    } catch (error) {
      console.error('Error fetching dashboard stats:', error);
    } finally {
      setLoading(false);
    }
  };

  if (loading) {
    return (
      <div className="flex items-center justify-center h-64">
        <div className="animate-spin rounded-full h-12 w-12 border-b-2 border-blue-600"></div>
      </div>
    );
  }

  return (
    <div className="space-y-8">
      <div>
        <h1 className="text-3xl font-bold text-gray-900">Dashboard</h1>
        <p className="mt-2 text-gray-600">Welcome back! Here's what's happening today.</p>
      </div>

      {/* Stats Cards */}
      <div className="grid grid-cols-1 md:grid-cols-2 lg:grid-cols-4 gap-6">
        <div className="bg-white rounded-xl shadow-sm p-6 border border-gray-100">
          <div className="flex items-center">
            <div className="p-2 bg-green-100 rounded-lg">
              <DollarSign className="h-6 w-6 text-green-600" />
            </div>
            <div className="ml-4">
              <p className="text-sm font-medium text-gray-600">Today's Sales</p>
              <p className="text-2xl font-bold text-gray-900">
                KSh {stats?.todaySales.toLocaleString() || '0'}
              </p>
            </div>
          </div>
        </div>

        <div className="bg-white rounded-xl shadow-sm p-6 border border-gray-100">
          <div className="flex items-center">
            <div className="p-2 bg-blue-100 rounded-lg">
              <ShoppingCart className="h-6 w-6 text-blue-600" />
            </div>
            <div className="ml-4">
              <p className="text-sm font-medium text-gray-600">Transactions</p>
              <p className="text-2xl font-bold text-gray-900">
                {stats?.todayTransactions || 0}
              </p>
            </div>
          </div>
        </div>

        <div className="bg-white rounded-xl shadow-sm p-6 border border-gray-100">
          <div className="flex items-center">
            <div className="p-2 bg-purple-100 rounded-lg">
              <TrendingUp className="h-6 w-6 text-purple-600" />
            </div>
            <div className="ml-4">
              <p className="text-sm font-medium text-gray-600">Avg. Transaction</p>
              <p className="text-2xl font-bold text-gray-900">
                KSh {stats?.todayTransactions ? Math.round(stats.todaySales / stats.todayTransactions).toLocaleString() : '0'}
              </p>
            </div>
          </div>
        </div>

        <div className="bg-white rounded-xl shadow-sm p-6 border border-gray-100">
          <div className="flex items-center">
            <div className="p-2 bg-orange-100 rounded-lg">
              <Users className="h-6 w-6 text-orange-600" />
            </div>
            <div className="ml-4">
              <p className="text-sm font-medium text-gray-600">Active Cashiers</p>
              <p className="text-2xl font-bold text-gray-900">
                {stats?.cashierPerformance.length || 0}
              </p>
            </div>
          </div>
        </div>
      </div>

      {/* Charts and Tables */}
      <div className="grid grid-cols-1 lg:grid-cols-2 gap-8">
        {/* Top Services */}
        <div className="bg-white rounded-xl shadow-sm border border-gray-100">
          <div className="p-6 border-b border-gray-100">
            <h3 className="text-lg font-semibold text-gray-900">Top Services Today</h3>
          </div>
          <div className="p-6">
            {stats?.topServices.length ? (
              <div className="space-y-4">
                {stats.topServices.map((service, index) => (
                  <div key={service.service_name} className="flex items-center justify-between">
                    <div className="flex items-center space-x-3">
                      <div className="w-8 h-8 bg-blue-100 rounded-full flex items-center justify-center">
                        <span className="text-sm font-medium text-blue-600">{index + 1}</span>
                      </div>
                      <div>
                        <p className="font-medium text-gray-900">{service.service_name}</p>
                        <p className="text-sm text-gray-500">{service.transaction_count} transactions</p>
                      </div>
                    </div>
                    <p className="font-semibold text-gray-900">
                      KSh {service.total_sales.toLocaleString()}
                    </p>
                  </div>
                ))}
              </div>
            ) : (
              <p className="text-gray-500 text-center py-8">No sales data available</p>
            )}
          </div>
        </div>

        {/* Cashier Performance */}
        <div className="bg-white rounded-xl shadow-sm border border-gray-100">
          <div className="p-6 border-b border-gray-100">
            <h3 className="text-lg font-semibold text-gray-900">Cashier Performance Today</h3>
          </div>
          <div className="p-6">
            {stats?.cashierPerformance.length ? (
              <div className="space-y-4">
                {stats.cashierPerformance.map((cashier, index) => (
                  <div key={cashier.cashier_name} className="flex items-center justify-between">
                    <div className="flex items-center space-x-3">
                      <div className="w-8 h-8 bg-green-100 rounded-full flex items-center justify-center">
                        <span className="text-sm font-medium text-green-600">{index + 1}</span>
                      </div>
                      <div>
                        <p className="font-medium text-gray-900">{cashier.cashier_name}</p>
                        <p className="text-sm text-gray-500">{cashier.transaction_count} transactions</p>
                      </div>
                    </div>
                    <p className="font-semibold text-gray-900">
                      KSh {cashier.total_sales.toLocaleString()}
                    </p>
                  </div>
                ))}
              </div>
            ) : (
              <p className="text-gray-500 text-center py-8">No sales data available</p>
            )}
          </div>
        </div>
      </div>
    </div>
  );
};