<?php
require_once 'config/config.php';
require_once 'config/database.php';
require_once 'includes/auth.php';
require_once 'includes/functions.php';

$auth->requireAdmin();

$pageTitle = 'System Settings';

// Handle form submission
if ($_SERVER['REQUEST_METHOD'] == 'POST') {
    if (!$auth->validateCSRFToken($_POST['csrf_token'] ?? '')) {
        die('CSRF token validation failed');
    }

    $shopName = sanitizeInput($_POST['shop_name']);
    $address = sanitizeInput($_POST['address']);
    $phoneNumber = sanitizeInput($_POST['phone_number']);
    $email = sanitizeInput($_POST['email']);
    $mpesaTillNumber = sanitizeInput($_POST['mpesa_till_number']);
    $taxRate = floatval($_POST['tax_rate']);
    $receiptHeader = sanitizeInput($_POST['receipt_header']);
    $receiptFooter = sanitizeInput($_POST['receipt_footer']);
    $logoUrl = sanitizeInput($_POST['logo_url']);

    if (empty($shopName) || empty($address) || empty($phoneNumber)) {
        $error = 'Please fill in all required fields';
    } elseif (!empty($email) && !validateEmail($email)) {
        $error = 'Please enter a valid email address';
    } elseif ($taxRate < 0 || $taxRate > 100) {
        $error = 'Tax rate must be between 0 and 100';
    } else {
        try {
            // Check if settings exist
            $stmt = $db->prepare("SELECT id FROM shop_settings LIMIT 1");
            $stmt->execute();
            $existingSettings = $stmt->fetch();

            $settingsData = [
                $shopName,
                $address,
                $phoneNumber,
                $email ?: null,
                $mpesaTillNumber ?: null,
                $taxRate,
                $receiptHeader ?: null,
                $receiptFooter ?: null,
                $logoUrl ?: null
            ];

            if ($existingSettings) {
                // Update existing settings
                $stmt = $db->prepare("
                    UPDATE shop_settings 
                    SET shop_name = ?, address = ?, phone_number = ?, email = ?, 
                        mpesa_till_number = ?, tax_rate = ?, receipt_header = ?, 
                        receipt_footer = ?, logo_url = ?
                    WHERE id = ?
                ");
                $settingsData[] = $existingSettings['id'];
                $stmt->execute($settingsData);
            } else {
                // Insert new settings
                $stmt = $db->prepare("
                    INSERT INTO shop_settings (
                        shop_name, address, phone_number, email, mpesa_till_number,
                        tax_rate, receipt_header, receipt_footer, logo_url
                    ) VALUES (?, ?, ?, ?, ?, ?, ?, ?, ?)
                ");
                $stmt->execute($settingsData);
            }

            $success = 'Settings saved successfully';
        } catch (Exception $e) {
            logError("Error saving settings: " . $e->getMessage());
            $error = 'Failed to save settings';
        }
    }
}

// Get current settings
$settings = getShopSettings($db);

include 'includes/header.php';
?>

<div class="space-y-8">
    <div>
        <h1 class="text-3xl font-bold text-gray-900">System Settings</h1>
        <p class="mt-2 text-gray-600">Configure your cyber shop details and preferences</p>
    </div>

    <?php if (isset($error)): ?>
    <div class="bg-red-50 border border-red-200 text-red-700 px-4 py-3 rounded-lg">
        <i class="fas fa-exclamation-circle mr-2"></i>
        <?php echo htmlspecialchars($error); ?>
    </div>
    <?php endif; ?>

    <?php if (isset($success)): ?>
    <div class="bg-green-50 border border-green-200 text-green-700 px-4 py-3 rounded-lg">
        <i class="fas fa-check-circle mr-2"></i>
        <?php echo htmlspecialchars($success); ?>
    </div>
    <?php endif; ?>

    <form method="POST" class="space-y-8">
        <input type="hidden" name="csrf_token" value="<?php echo $auth->generateCSRFToken(); ?>">

        <!-- Shop Information -->
        <div class="bg-white rounded-xl shadow-sm border border-gray-100 p-6">
            <div class="flex items-center space-x-2 mb-6">
                <i class="fas fa-building text-blue-600"></i>
                <h2 class="text-lg font-semibold text-gray-900">Shop Information</h2>
            </div>
            
            <div class="grid grid-cols-1 md:grid-cols-2 gap-6">
                <div>
                    <label class="block text-sm font-medium text-gray-700 mb-2">
                        Shop Name *
                    </label>
                    <input
                        type="text"
                        name="shop_name"
                        required
                        value="<?php echo htmlspecialchars($settings['shop_name'] ?? ''); ?>"
                        class="w-full px-3 py-2 border border-gray-300 rounded-lg focus:ring-2 focus:ring-blue-500 focus:border-blue-500"
                        placeholder="e.g., Digital Hub Cyber"
                    />
                </div>

                <div>
                    <label class="block text-sm font-medium text-gray-700 mb-2">
                        Phone Number *
                    </label>
                    <input
                        type="tel"
                        name="phone_number"
                        required
                        value="<?php echo htmlspecialchars($settings['phone_number'] ?? ''); ?>"
                        class="w-full px-3 py-2 border border-gray-300 rounded-lg focus:ring-2 focus:ring-blue-500 focus:border-blue-500"
                        placeholder="e.g., +254 700 000 000"
                    />
                </div>

                <div class="md:col-span-2">
                    <label class="block text-sm font-medium text-gray-700 mb-2">
                        Address *
                    </label>
                    <textarea
                        name="address"
                        required
                        rows="3"
                        class="w-full px-3 py-2 border border-gray-300 rounded-lg focus:ring-2 focus:ring-blue-500 focus:border-blue-500"
                        placeholder="Enter shop address"
                    ><?php echo htmlspecialchars($settings['address'] ?? ''); ?></textarea>
                </div>

                <div>
                    <label class="block text-sm font-medium text-gray-700 mb-2">
                        Email
                    </label>
                    <input
                        type="email"
                        name="email"
                        value="<?php echo htmlspecialchars($settings['email'] ?? ''); ?>"
                        class="w-full px-3 py-2 border border-gray-300 rounded-lg focus:ring-2 focus:ring-blue-500 focus:border-blue-500"
                        placeholder="shop@example.com"
                    />
                </div>

                <div>
                    <label class="block text-sm font-medium text-gray-700 mb-2">
                        M-Pesa Till Number
                    </label>
                    <input
                        type="text"
                        name="mpesa_till_number"
                        value="<?php echo htmlspecialchars($settings['mpesa_till_number'] ?? ''); ?>"
                        class="w-full px-3 py-2 border border-gray-300 rounded-lg focus:ring-2 focus:ring-blue-500 focus:border-blue-500"
                        placeholder="e.g., 123456"
                    />
                </div>
            </div>
        </div>

        <!-- Financial Settings -->
        <div class="bg-white rounded-xl shadow-sm border border-gray-100 p-6">
            <div class="flex items-center space-x-2 mb-6">
                <i class="fas fa-credit-card text-blue-600"></i>
                <h2 class="text-lg font-semibold text-gray-900">Financial Settings</h2>
            </div>
            
            <div>
                <label class="block text-sm font-medium text-gray-700 mb-2">
                    Tax Rate (%)
                </label>
                <input
                    type="number"
                    name="tax_rate"
                    min="0"
                    max="100"
                    step="0.01"
                    value="<?php echo $settings['tax_rate'] ?? 16; ?>"
                    class="w-full px-3 py-2 border border-gray-300 rounded-lg focus:ring-2 focus:ring-blue-500 focus:border-blue-500"
                    placeholder="16"
                />
                <p class="text-sm text-gray-500 mt-1">VAT rate applied to all sales</p>
            </div>
        </div>

        <!-- Receipt Settings -->
        <div class="bg-white rounded-xl shadow-sm border border-gray-100 p-6">
            <div class="flex items-center space-x-2 mb-6">
                <i class="fas fa-receipt text-blue-600"></i>
                <h2 class="text-lg font-semibold text-gray-900">Receipt Settings</h2>
            </div>
            
            <div class="space-y-4">
                <div>
                    <label class="block text-sm font-medium text-gray-700 mb-2">
                        Receipt Header
                    </label>
                    <textarea
                        name="receipt_header"
                        rows="2"
                        class="w-full px-3 py-2 border border-gray-300 rounded-lg focus:ring-2 focus:ring-blue-500 focus:border-blue-500"
                        placeholder="Welcome to our cyber shop!"
                    ><?php echo htmlspecialchars($settings['receipt_header'] ?? ''); ?></textarea>
                </div>

                <div>
                    <label class="block text-sm font-medium text-gray-700 mb-2">
                        Receipt Footer
                    </label>
                    <textarea
                        name="receipt_footer"
                        rows="2"
                        class="w-full px-3 py-2 border border-gray-300 rounded-lg focus:ring-2 focus:ring-blue-500 focus:border-blue-500"
                        placeholder="Thank you for your business!"
                    ><?php echo htmlspecialchars($settings['receipt_footer'] ?? ''); ?></textarea>
                </div>

                <div>
                    <label class="block text-sm font-medium text-gray-700 mb-2">
                        Logo URL
                    </label>
                    <input
                        type="url"
                        name="logo_url"
                        value="<?php echo htmlspecialchars($settings['logo_url'] ?? ''); ?>"
                        class="w-full px-3 py-2 border border-gray-300 rounded-lg focus:ring-2 focus:ring-blue-500 focus:border-blue-500"
                        placeholder="https://example.com/logo.png"
                    />
                </div>
            </div>
        </div>

        <div class="flex justify-end">
            <button
                type="submit"
                class="bg-blue-600 text-white px-6 py-2 rounded-lg hover:bg-blue-700 flex items-center space-x-2 transition-colors"
            >
                <i class="fas fa-save"></i>
                <span>Save Settings</span>
            </button>
        </div>
    </form>
</div>

</main>
</body>
</html>