<?php
require_once 'config/config.php';
require_once 'config/database.php';
require_once 'includes/auth.php';
require_once 'includes/functions.php';

$auth->requireAdmin();

$pageTitle = 'M-Pesa Integration';

// Handle form submission
if ($_SERVER['REQUEST_METHOD'] == 'POST') {
    if (!$auth->validateCSRFToken($_POST['csrf_token'] ?? '')) {
        die('CSRF token validation failed');
    }

    $consumerKey = sanitizeInput($_POST['consumer_key']);
    $consumerSecret = sanitizeInput($_POST['consumer_secret']);
    $passkey = sanitizeInput($_POST['passkey']);
    $shortcode = sanitizeInput($_POST['shortcode']);
    $environment = sanitizeInput($_POST['environment']);

    try {
        // Save M-Pesa configuration (in real app, encrypt these values)
        $stmt = $db->prepare("
            INSERT INTO mpesa_config (consumer_key, consumer_secret, passkey, shortcode, environment)
            VALUES (?, ?, ?, ?, ?)
            ON DUPLICATE KEY UPDATE
            consumer_key = VALUES(consumer_key),
            consumer_secret = VALUES(consumer_secret),
            passkey = VALUES(passkey),
            shortcode = VALUES(shortcode),
            environment = VALUES(environment)
        ");
        $stmt->execute([$consumerKey, $consumerSecret, $passkey, $shortcode, $environment]);
        $success = 'M-Pesa configuration saved successfully';
    } catch (Exception $e) {
        logError("Error saving M-Pesa config: " . $e->getMessage());
        $error = 'Failed to save M-Pesa configuration';
    }
}

// Get current M-Pesa configuration
try {
    $stmt = $db->prepare("SELECT * FROM mpesa_config ORDER BY id DESC LIMIT 1");
    $stmt->execute();
    $mpesaConfig = $stmt->fetch();
} catch (Exception $e) {
    $mpesaConfig = null;
}

include 'includes/header.php';
?>

<div class="space-y-8">
    <div>
        <h1 class="text-3xl font-bold text-gray-900">M-Pesa Integration</h1>
        <p class="mt-2 text-gray-600">Configure M-Pesa API for automatic payment processing</p>
    </div>

    <?php if (isset($error)): ?>
    <div class="bg-red-50 border border-red-200 text-red-700 px-4 py-3 rounded-lg">
        <i class="fas fa-exclamation-circle mr-2"></i>
        <?php echo htmlspecialchars($error); ?>
    </div>
    <?php endif; ?>

    <?php if (isset($success)): ?>
    <div class="bg-green-50 border border-green-200 text-green-700 px-4 py-3 rounded-lg">
        <i class="fas fa-check-circle mr-2"></i>
        <?php echo htmlspecialchars($success); ?>
    </div>
    <?php endif; ?>

    <!-- Setup Instructions -->
    <div class="bg-blue-50 border border-blue-200 rounded-xl p-6">
        <div class="flex items-start">
            <i class="fas fa-info-circle text-blue-600 mt-1 mr-3"></i>
            <div>
                <h3 class="text-sm font-medium text-blue-800 mb-2">M-Pesa API Setup Instructions</h3>
                <ol class="text-sm text-blue-700 space-y-1">
                    <li>1. Register for M-Pesa API at <a href="https://developer.safaricom.co.ke" target="_blank" class="underline">developer.safaricom.co.ke</a></li>
                    <li>2. Create a new app and get your Consumer Key and Consumer Secret</li>
                    <li>3. Get your Business Shortcode and Passkey from Safaricom</li>
                    <li>4. Test in Sandbox environment first, then switch to Production</li>
                    <li>5. Configure the settings below with your API credentials</li>
                </ol>
            </div>
        </div>
    </div>

    <!-- Configuration Form -->
    <div class="bg-white rounded-xl shadow-sm border border-gray-100 p-6">
        <h2 class="text-lg font-semibold text-gray-900 mb-6">API Configuration</h2>
        
        <form method="POST" class="space-y-6">
            <input type="hidden" name="csrf_token" value="<?php echo $auth->generateCSRFToken(); ?>">

            <div class="grid grid-cols-1 md:grid-cols-2 gap-6">
                <div>
                    <label class="block text-sm font-medium text-gray-700 mb-2">
                        Consumer Key *
                    </label>
                    <input
                        type="text"
                        name="consumer_key"
                        required
                        value="<?php echo htmlspecialchars($mpesaConfig['consumer_key'] ?? ''); ?>"
                        class="w-full px-3 py-2 border border-gray-300 rounded-lg focus:ring-2 focus:ring-blue-500 focus:border-blue-500"
                        placeholder="Your M-Pesa Consumer Key"
                    />
                </div>

                <div>
                    <label class="block text-sm font-medium text-gray-700 mb-2">
                        Consumer Secret *
                    </label>
                    <input
                        type="password"
                        name="consumer_secret"
                        required
                        value="<?php echo htmlspecialchars($mpesaConfig['consumer_secret'] ?? ''); ?>"
                        class="w-full px-3 py-2 border border-gray-300 rounded-lg focus:ring-2 focus:ring-blue-500 focus:border-blue-500"
                        placeholder="Your M-Pesa Consumer Secret"
                    />
                </div>

                <div>
                    <label class="block text-sm font-medium text-gray-700 mb-2">
                        Business Shortcode *
                    </label>
                    <input
                        type="text"
                        name="shortcode"
                        required
                        value="<?php echo htmlspecialchars($mpesaConfig['shortcode'] ?? ''); ?>"
                        class="w-full px-3 py-2 border border-gray-300 rounded-lg focus:ring-2 focus:ring-blue-500 focus:border-blue-500"
                        placeholder="e.g., 174379"
                    />
                </div>

                <div>
                    <label class="block text-sm font-medium text-gray-700 mb-2">
                        Passkey *
                    </label>
                    <input
                        type="password"
                        name="passkey"
                        required
                        value="<?php echo htmlspecialchars($mpesaConfig['passkey'] ?? ''); ?>"
                        class="w-full px-3 py-2 border border-gray-300 rounded-lg focus:ring-2 focus:ring-blue-500 focus:border-blue-500"
                        placeholder="Your M-Pesa Passkey"
                    />
                </div>

                <div class="md:col-span-2">
                    <label class="block text-sm font-medium text-gray-700 mb-2">
                        Environment *
                    </label>
                    <select
                        name="environment"
                        class="w-full px-3 py-2 border border-gray-300 rounded-lg focus:ring-2 focus:ring-blue-500 focus:border-blue-500"
                    >
                        <option value="sandbox" <?php echo ($mpesaConfig['environment'] ?? '') === 'sandbox' ? 'selected' : ''; ?>>
                            Sandbox (Testing)
                        </option>
                        <option value="production" <?php echo ($mpesaConfig['environment'] ?? '') === 'production' ? 'selected' : ''; ?>>
                            Production (Live)
                        </option>
                    </select>
                    <p class="text-xs text-gray-500 mt-1">Use Sandbox for testing, Production for live transactions</p>
                </div>
            </div>

            <div class="flex justify-end">
                <button
                    type="submit"
                    class="bg-blue-600 text-white px-6 py-2 rounded-lg hover:bg-blue-700 flex items-center space-x-2 transition-colors"
                >
                    <i class="fas fa-save"></i>
                    <span>Save Configuration</span>
                </button>
            </div>
        </form>
    </div>

    <!-- Test M-Pesa -->
    <div class="bg-white rounded-xl shadow-sm border border-gray-100 p-6">
        <h2 class="text-lg font-semibold text-gray-900 mb-6">Test M-Pesa Integration</h2>
        
        <div class="grid grid-cols-1 md:grid-cols-2 gap-6">
            <div>
                <label class="block text-sm font-medium text-gray-700 mb-2">
                    Phone Number
                </label>
                <input
                    type="tel"
                    id="testPhone"
                    placeholder="254700000000"
                    class="w-full px-3 py-2 border border-gray-300 rounded-lg focus:ring-2 focus:ring-blue-500 focus:border-blue-500"
                />
            </div>

            <div>
                <label class="block text-sm font-medium text-gray-700 mb-2">
                    Amount (KSh)
                </label>
                <input
                    type="number"
                    id="testAmount"
                    min="1"
                    placeholder="100"
                    class="w-full px-3 py-2 border border-gray-300 rounded-lg focus:ring-2 focus:ring-blue-500 focus:border-blue-500"
                />
            </div>
        </div>

        <div class="mt-6">
            <button
                onclick="testMpesaPayment()"
                class="bg-green-600 text-white px-4 py-2 rounded-lg hover:bg-green-700 flex items-center space-x-2 transition-colors"
            >
                <i class="fas fa-mobile-alt"></i>
                <span>Test STK Push</span>
            </button>
        </div>

        <div id="testResult" class="mt-4 hidden">
            <!-- Test results will appear here -->
        </div>
    </div>

    <!-- Transaction Log -->
    <div class="bg-white rounded-xl shadow-sm border border-gray-100">
        <div class="p-6 border-b border-gray-100">
            <h2 class="text-lg font-semibold text-gray-900">Recent M-Pesa Transactions</h2>
        </div>
        <div class="p-6">
            <div class="text-center py-8 text-gray-500">
                <i class="fas fa-mobile-alt text-4xl mb-4"></i>
                <p>M-Pesa transaction log will appear here</p>
                <p class="text-sm text-gray-400">Configure API credentials and start processing payments</p>
            </div>
        </div>
    </div>
</div>

<script>
function testMpesaPayment() {
    const phone = document.getElementById('testPhone').value;
    const amount = document.getElementById('testAmount').value;
    const resultDiv = document.getElementById('testResult');

    if (!phone || !amount) {
        alert('Please enter phone number and amount');
        return;
    }

    resultDiv.innerHTML = '<div class="bg-blue-50 border border-blue-200 text-blue-700 px-4 py-3 rounded-lg"><i class="fas fa-spinner fa-spin mr-2"></i>Sending payment request...</div>';
    resultDiv.classList.remove('hidden');

    fetch('api/mpesa.php', {
        method: 'POST',
        headers: {
            'Content-Type': 'application/x-www-form-urlencoded',
        },
        body: `action=initiate_payment&phone_number=${phone}&amount=${amount}&reference=TEST`
    })
    .then(response => response.json())
    .then(data => {
        if (data.success) {
            resultDiv.innerHTML = `
                <div class="bg-green-50 border border-green-200 text-green-700 px-4 py-3 rounded-lg">
                    <i class="fas fa-check-circle mr-2"></i>
                    ${data.message}
                    <br><small>Transaction ID: ${data.transaction_id}</small>
                </div>
            `;
        } else {
            resultDiv.innerHTML = `
                <div class="bg-red-50 border border-red-200 text-red-700 px-4 py-3 rounded-lg">
                    <i class="fas fa-exclamation-circle mr-2"></i>
                    ${data.message}
                </div>
            `;
        }
    })
    .catch(error => {
        resultDiv.innerHTML = `
            <div class="bg-red-50 border border-red-200 text-red-700 px-4 py-3 rounded-lg">
                <i class="fas fa-exclamation-circle mr-2"></i>
                Error testing M-Pesa integration
            </div>
        `;
    });
}
</script>

</main>
</body>
</html>